"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ContentStream = void 0;
exports.getReadableContentStream = getReadableContentStream;
exports.getWritableContentStream = getWritableContentStream;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _crypto = require("crypto");
var _cbor = require("@kbn/cbor");
var _elasticsearch = require("@elastic/elasticsearch");
var _configSchema = require("@kbn/config-schema");
var _lodash = require("lodash");
var _stream = require("stream");
var _util = require("util");
var _utils = require("../../../../file_client/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class ContentStream extends _stream.Duplex {
  constructor(client, id, index, logger, parameters = {}, indexIsAlias = false) {
    super();
    (0, _defineProperty2.default)(this, "buffers", []);
    (0, _defineProperty2.default)(this, "bytesBuffered", 0);
    (0, _defineProperty2.default)(this, "bytesRead", 0);
    (0, _defineProperty2.default)(this, "chunksRead", 0);
    (0, _defineProperty2.default)(this, "chunksWritten", 0);
    (0, _defineProperty2.default)(this, "maxChunkSize", void 0);
    (0, _defineProperty2.default)(this, "parameters", void 0);
    /**
     * The number of bytes written so far.
     * Does not include data that is still queued for writing.
     */
    (0, _defineProperty2.default)(this, "bytesWritten", 0);
    /**
     * Holds a reference to the last written chunk without actually writing it to ES.
     *
     * This enables us to reliably determine what the real last chunk is at the cost
     * of holding, at most, 2 full chunks in memory.
     */
    (0, _defineProperty2.default)(this, "indexRequestBuffer", void 0);
    this.client = client;
    this.id = id;
    this.index = index;
    this.logger = logger;
    this.indexIsAlias = indexIsAlias;
    this.parameters = (0, _lodash.defaults)(parameters, {
      encoding: 'base64',
      size: -1,
      maxChunkSize: '4mb'
    });
  }
  getMaxContentSize() {
    return _configSchema.ByteSizeValue.parse(this.parameters.maxChunkSize).getValueInBytes();
  }
  getMaxChunkSize() {
    if (!this.maxChunkSize) {
      this.maxChunkSize = this.getMaxContentSize();
      this.logger.debug(`Chunk size is ${this.maxChunkSize} bytes.`);
    }
    return this.maxChunkSize;
  }
  async getChunkRealIndex(id) {
    var _chunkDocMeta$hits$hi, _chunkDocMeta$hits$hi2;
    const chunkDocMeta = await this.client.search({
      index: this.index,
      body: {
        size: 1,
        query: {
          term: {
            _id: id
          }
        },
        _source: false // suppress the document content
      }
    });
    const docIndex = (_chunkDocMeta$hits$hi = chunkDocMeta.hits.hits) === null || _chunkDocMeta$hits$hi === void 0 ? void 0 : (_chunkDocMeta$hits$hi2 = _chunkDocMeta$hits$hi[0]) === null || _chunkDocMeta$hits$hi2 === void 0 ? void 0 : _chunkDocMeta$hits$hi2._index;
    if (!docIndex) {
      const err = new Error(`Unable to determine index for file chunk id [${id}] in index (alias) [${this.index}]`);
      this.logger.error(err);
      throw err;
    }
    return docIndex;
  }
  async readChunk() {
    if (!this.id) {
      throw new Error('No document ID provided. Cannot read chunk.');
    }
    const id = this.getChunkId(this.chunksRead);
    const chunkIndex = this.indexIsAlias ? await this.getChunkRealIndex(id) : this.index;
    this.logger.debug(`Reading chunk #${this.chunksRead} from index [${chunkIndex}]`);
    try {
      const stream = await this.client.get({
        id,
        index: chunkIndex,
        _source_includes: ['data', 'last']
      }, {
        asStream: true,
        // This tells the ES client to not process the response body in any way.
        headers: {
          accept: 'application/cbor'
        }
      });
      const chunks = [];
      for await (const chunk of stream) {
        chunks.push(chunk);
      }
      const buffer = Buffer.concat(chunks);
      const decodedChunkDoc = buffer.byteLength ? (0, _cbor.decode)(buffer) : undefined;

      // Because `asStream` was used in retrieving the document, errors are also not be processed
      // and thus are returned "as is", so we check to see if an ES error occurred while attempting
      // to retrieve the chunk.
      if (decodedChunkDoc && ('error' in decodedChunkDoc || !decodedChunkDoc.found)) {
        const err = new Error(`Failed to retrieve chunk id [${id}] from index [${chunkIndex}]`);
        this.logger.error(err);
        this.logger.error((0, _util.inspect)(decodedChunkDoc, {
          depth: 5
        }));
        throw err;
      }
      const source = decodedChunkDoc === null || decodedChunkDoc === void 0 ? void 0 : decodedChunkDoc._source;
      const dataBuffer = source === null || source === void 0 ? void 0 : source.data;
      return [dataBuffer !== null && dataBuffer !== void 0 && dataBuffer.byteLength ? dataBuffer : null, source === null || source === void 0 ? void 0 : source.last];
    } catch (e) {
      if (e instanceof _elasticsearch.errors.ResponseError && e.statusCode === 404) {
        const readingHeadChunk = this.chunksRead <= 0;
        if (this.isSizeUnknown() && !readingHeadChunk) {
          // Assume there is no more content to read.
          return [null];
        }
        if (readingHeadChunk) {
          this.logger.error(`File not found (id: ${this.getHeadChunkId()}).`);
        }
      }
      throw e;
    }
  }
  isSizeUnknown() {
    return this.parameters.size < 0;
  }
  isRead() {
    const {
      size
    } = this.parameters;
    if (size > 0) {
      return this.bytesRead >= size;
    }
    return false;
  }
  _read() {
    this.readChunk().then(([buffer, last]) => {
      if (!buffer) {
        this.logger.debug(`Chunk is empty.`);
        this.push(null);
        return;
      }
      this.push(buffer);
      this.chunksRead++;
      this.bytesRead += buffer.byteLength;
      if (this.isRead() || last) {
        this.logger.debug(`Read ${this.bytesRead} of ${this.parameters.size} bytes.`);
        this.push(null);
      }
    }).catch(err => this.destroy(err));
  }
  async removeChunks() {
    const bid = this.getId();
    this.logger.debug(`Clearing existing chunks for ${bid}`);
    await this.client.deleteByQuery({
      index: this.index,
      ignore_unavailable: true,
      query: {
        bool: {
          must: {
            match: {
              bid
            }
          }
        }
      }
    });
  }
  getId() {
    if (!this.id) {
      this.id = (0, _crypto.randomUUID)();
    }
    return this.id;
  }
  getHeadChunkId() {
    return `${this.getId()}.0`;
  }
  getChunkId(chunkNumber = 0) {
    return chunkNumber === 0 ? this.getHeadChunkId() : `${this.getId()}.${chunkNumber}`;
  }
  async indexChunk({
    bid,
    data,
    id,
    index
  }, last) {
    await this.client.index({
      id,
      index,
      op_type: 'create',
      document: (0, _cbor.encode)({
        data,
        bid,
        // Mark it as last?
        ...(last ? {
          last
        } : {}),
        // Add `@timestamp` for Index Alias/DS?
        ...(this.indexIsAlias ? {
          '@timestamp': new Date().toISOString()
        } : {})
      })
    }, {
      headers: {
        'content-type': 'application/cbor',
        accept: 'application/json'
      }
    }).catch(_utils.wrapErrorAndReThrow.withMessagePrefix('ContentStream.indexChunk(): '));
  }
  async writeChunk(data) {
    const chunkId = this.getChunkId(this.chunksWritten);
    if (!this.indexRequestBuffer) {
      this.indexRequestBuffer = {
        id: chunkId,
        index: this.index,
        data,
        bid: this.getId()
      };
      return;
    }
    this.logger.debug(`Writing chunk with ID "${this.indexRequestBuffer.id}".`);
    await this.indexChunk(this.indexRequestBuffer);
    // Hold a reference to the next buffer now that we indexed the previous one.
    this.indexRequestBuffer = {
      id: chunkId,
      index: this.index,
      data,
      bid: this.getId()
    };
  }
  async finalizeLastChunk() {
    if (!this.indexRequestBuffer) {
      return;
    }
    this.logger.debug(`Writing last chunk with id "${this.indexRequestBuffer.id}".`);
    await this.indexChunk(this.indexRequestBuffer, true);
    this.indexRequestBuffer = undefined;
  }
  async flush(size = this.bytesBuffered) {
    const buffersToFlush = [];
    let bytesToFlush = 0;

    /*
     Loop over each buffer, keeping track of how many bytes we have added
     to the array of buffers to be flushed. The array of buffers to be flushed
     contains buffers by reference, not copies. This avoids putting pressure on
     the CPU for copying buffers or for gc activity. Please profile performance
     with a large byte configuration and a large number of records (900k+)
     before changing this code.
    */
    while (this.buffers.length) {
      const remainder = size - bytesToFlush;
      if (remainder <= 0) {
        break;
      }
      const buffer = this.buffers.shift();
      const chunkedBuffer = buffer.slice(0, remainder);
      buffersToFlush.push(chunkedBuffer);
      bytesToFlush += chunkedBuffer.byteLength;
      if (buffer.byteLength > remainder) {
        this.buffers.unshift(buffer.slice(remainder));
      }
    }

    // We call Buffer.concat with the fewest number of buffers possible
    const chunk = Buffer.concat(buffersToFlush);
    if (!this.chunksWritten) {
      await this.removeChunks();
    }
    if (chunk.byteLength) {
      await this.writeChunk(chunk);
      this.chunksWritten++;
    }
    this.bytesWritten += chunk.byteLength;
    this.bytesBuffered -= bytesToFlush;
  }
  async flushAllFullChunks() {
    const maxChunkSize = this.getMaxChunkSize();
    while (this.bytesBuffered >= maxChunkSize && this.buffers.length) {
      await this.flush(maxChunkSize);
    }
  }
  _write(chunk, encoding, callback) {
    const buffer = Buffer.isBuffer(chunk) ? chunk : Buffer.from(chunk, encoding);
    this.bytesBuffered += buffer.byteLength;
    this.buffers.push(buffer);
    this.flushAllFullChunks().then(() => callback()).catch(callback);
  }
  _final(callback) {
    this.flush().then(() => this.finalizeLastChunk()).then(() => callback()).catch(callback);
  }

  /**
   * This ID can be used to retrieve or delete all of the file chunks but does
   * not necessarily correspond to an existing document.
   *
   * @note do not use this ID with anything other than a {@link ContentStream}
   * compatible implementation for reading blob-like structures from ES.
   *
   * @note When creating a new blob, this value may be undefined until the first
   * chunk is written.
   */
  getContentReferenceId() {
    return this.id;
  }
  getBytesWritten() {
    return this.bytesWritten;
  }
}
exports.ContentStream = ContentStream;
function getContentStream({
  client,
  id,
  index,
  logger,
  parameters,
  indexIsAlias = false
}) {
  return new ContentStream(client, id, index, logger, parameters, indexIsAlias);
}
function getWritableContentStream(args) {
  return getContentStream(args);
}
function getReadableContentStream(args) {
  return getContentStream(args);
}