"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ignoreErrorsMap = void 0;
exports.validateQuery = validateQuery;
exports.validateSources = validateSources;
var _uniqBy = _interopRequireDefault(require("lodash/uniqBy"));
var _esqlAst = require("@kbn/esql-ast");
var _helpers = require("../shared/helpers");
var _variables = require("../shared/variables");
var _errors = require("./errors");
var _resources = require("./resources");
var _helpers2 = require("./helpers");
var _constants = require("../shared/constants");
var _esql_types = require("../shared/esql_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const NO_MESSAGE = [];
function validateFunctionLiteralArg(astFunction, actualArg, argDef, references, parentCommand) {
  const messages = [];
  if ((0, _helpers.isLiteralItem)(actualArg)) {
    if (actualArg.literalType === 'keyword' && argDef.acceptedValues && (0, _helpers.isValidLiteralOption)(actualArg, argDef)) {
      var _argDef$acceptedValue;
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unsupportedLiteralOption',
        values: {
          name: astFunction.name,
          value: actualArg.value,
          supportedOptions: (_argDef$acceptedValue = argDef.acceptedValues) === null || _argDef$acceptedValue === void 0 ? void 0 : _argDef$acceptedValue.map(option => `"${option}"`).join(', ')
        },
        locations: actualArg.location
      }));
    }
    if (!(0, _helpers.checkFunctionArgMatchesDefinition)(actualArg, argDef, references, parentCommand)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentType',
        values: {
          name: astFunction.name,
          argType: argDef.type,
          value: actualArg.text,
          givenType: actualArg.literalType
        },
        locations: actualArg.location
      }));
    }
  }
  if ((0, _helpers.isTimeIntervalItem)(actualArg)) {
    // check first if it's a valid interval string
    if (!(0, _helpers.inKnownTimeInterval)(actualArg.unit)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unknownInterval',
        values: {
          value: actualArg.unit
        },
        locations: actualArg.location
      }));
    } else {
      if (!(0, _helpers.checkFunctionArgMatchesDefinition)(actualArg, argDef, references, parentCommand)) {
        messages.push((0, _errors.getMessageFromId)({
          messageId: 'wrongArgumentType',
          values: {
            name: astFunction.name,
            argType: argDef.type,
            value: actualArg.name,
            givenType: 'duration'
          },
          locations: actualArg.location
        }));
      }
    }
  }
  return messages;
}
function validateInlineCastArg(astFunction, arg, parameterDefinition, references, parentCommand) {
  if (!(0, _helpers.isInlineCastItem)(arg)) {
    return [];
  }
  if (!(0, _helpers.checkFunctionArgMatchesDefinition)(arg, parameterDefinition, references, parentCommand)) {
    return [(0, _errors.getMessageFromId)({
      messageId: 'wrongArgumentType',
      values: {
        name: astFunction.name,
        argType: parameterDefinition.type,
        value: arg.text,
        givenType: arg.castType
      },
      locations: arg.location
    })];
  }
  return [];
}
function validateNestedFunctionArg(astFunction, actualArg, parameterDefinition, references, parentCommand) {
  const messages = [];
  if ((0, _helpers.isFunctionItem)(actualArg) &&
  // no need to check the reason here, it is checked already above
  (0, _helpers.isSupportedFunction)(actualArg.name, parentCommand).supported) {
    // The isSupported check ensure the definition exists
    const argFn = (0, _helpers.getFunctionDefinition)(actualArg.name);
    const fnDef = (0, _helpers.getFunctionDefinition)(astFunction.name);
    // no nestying criteria should be enforced only for same type function
    if (fnDef.type === 'agg' && argFn.type === 'agg') {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'noNestedArgumentSupport',
        values: {
          name: actualArg.text,
          argType: argFn.signatures[0].returnType
        },
        locations: actualArg.location
      }));
    }
    if (!(0, _helpers.checkFunctionArgMatchesDefinition)(actualArg, parameterDefinition, references, parentCommand)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentType',
        values: {
          name: astFunction.name,
          argType: parameterDefinition.type,
          value: actualArg.text,
          givenType: argFn.signatures[0].returnType
        },
        locations: actualArg.location
      }));
    }
  }
  return messages;
}
function validateFunctionColumnArg(astFunction, actualArg, parameterDefinition, references, parentCommand) {
  const messages = [];
  if (!((0, _helpers.isColumnItem)(actualArg) || (0, _esqlAst.isIdentifier)(actualArg))) {
    return messages;
  }
  const columnName = (0, _helpers.getQuotedColumnName)(actualArg);
  const columnExists = (0, _helpers.getColumnExists)(actualArg, references);
  if (parameterDefinition.constantOnly) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'expectedConstant',
      values: {
        fn: astFunction.name,
        given: columnName
      },
      locations: actualArg.location
    }));
    return messages;
  }
  if (!columnExists) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unknownColumn',
      values: {
        name: actualArg.name
      },
      locations: actualArg.location
    }));
    return messages;
  }
  if (actualArg.name === '*') {
    // if function does not support wildcards return a specific error
    if (!('supportsWildcard' in parameterDefinition) || !parameterDefinition.supportsWildcard) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'noWildcardSupportAsArg',
        values: {
          name: astFunction.name
        },
        locations: actualArg.location
      }));
    }
    return messages;
  }
  if (!(0, _helpers.checkFunctionArgMatchesDefinition)(actualArg, parameterDefinition, references, parentCommand)) {
    const columnHit = (0, _helpers.getColumnForASTNode)(actualArg, references);
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'wrongArgumentType',
      values: {
        name: astFunction.name,
        argType: parameterDefinition.type,
        value: actualArg.name,
        givenType: columnHit.type
      },
      locations: actualArg.location
    }));
  }
  return messages;
}
function removeInlineCasts(arg) {
  if ((0, _helpers.isInlineCastItem)(arg)) {
    return removeInlineCasts(arg.value);
  }
  return arg;
}
function validateIfHasUnsupportedCommandPrior(fn, parentAst = [], unsupportedCommands, currentCommandIndex) {
  if (currentCommandIndex === undefined) {
    return NO_MESSAGE;
  }
  const unsupportedCommandsPrior = parentAst.filter((cmd, idx) => idx <= currentCommandIndex && unsupportedCommands.has(cmd.name));
  if (unsupportedCommandsPrior.length > 0) {
    return [(0, _errors.getMessageFromId)({
      messageId: 'fnUnsupportedAfterCommand',
      values: {
        function: fn.name.toUpperCase(),
        command: unsupportedCommandsPrior[0].name.toUpperCase()
      },
      locations: fn.location
    })];
  }
  return NO_MESSAGE;
}
const validateMatchFunction = ({
  fn,
  parentCommand,
  parentOption,
  references,
  forceConstantOnly = false,
  isNested,
  parentAst,
  currentCommandIndex
}) => {
  if (fn.name === 'match') {
    if (parentCommand !== 'where') {
      return [(0, _errors.getMessageFromId)({
        messageId: 'onlyWhereCommandSupported',
        values: {
          fn: fn.name
        },
        locations: fn.location
      })];
    }
    return validateIfHasUnsupportedCommandPrior(fn, parentAst, _constants.UNSUPPORTED_COMMANDS_BEFORE_MATCH, currentCommandIndex);
  }
  return NO_MESSAGE;
};
const validateQSTRFunction = ({
  fn,
  parentCommand,
  parentOption,
  references,
  forceConstantOnly = false,
  isNested,
  parentAst,
  currentCommandIndex
}) => {
  if (fn.name === 'qstr') {
    return validateIfHasUnsupportedCommandPrior(fn, parentAst, _constants.UNSUPPORTED_COMMANDS_BEFORE_QSTR, currentCommandIndex);
  }
  return NO_MESSAGE;
};
const textSearchFunctionsValidators = {
  match: validateMatchFunction,
  qstr: validateQSTRFunction
};
function validateFunction({
  fn,
  parentCommand,
  parentOption,
  references,
  forceConstantOnly = false,
  isNested,
  parentAst,
  currentCommandIndex
}) {
  const messages = [];
  if (fn.incomplete) {
    return messages;
  }
  if ((0, _helpers.isFunctionOperatorParam)(fn)) {
    return messages;
  }
  const fnDefinition = (0, _helpers.getFunctionDefinition)(fn.name);
  const isFnSupported = (0, _helpers.isSupportedFunction)(fn.name, parentCommand, parentOption);
  if (typeof textSearchFunctionsValidators[fn.name] === 'function') {
    const validator = textSearchFunctionsValidators[fn.name];
    messages.push(...validator({
      fn,
      parentCommand,
      parentOption,
      references,
      isNested,
      parentAst,
      currentCommandIndex
    }));
  }
  if (!isFnSupported.supported) {
    if (isFnSupported.reason === 'unknownFunction') {
      messages.push(_errors.errors.unknownFunction(fn));
    }
    // for nested functions skip this check and make the nested check fail later on
    if (isFnSupported.reason === 'unsupportedFunction' && !isNested) {
      messages.push(parentOption ? (0, _errors.getMessageFromId)({
        messageId: 'unsupportedFunctionForCommandOption',
        values: {
          name: fn.name,
          command: parentCommand.toUpperCase(),
          option: parentOption.toUpperCase()
        },
        locations: fn.location
      }) : (0, _errors.getMessageFromId)({
        messageId: 'unsupportedFunctionForCommand',
        values: {
          name: fn.name,
          command: parentCommand.toUpperCase()
        },
        locations: fn.location
      }));
    }
    if (messages.length) {
      return messages;
    }
  }
  const matchingSignatures = (0, _helpers.getSignaturesWithMatchingArity)(fnDefinition, fn);
  if (!matchingSignatures.length) {
    const {
      max,
      min
    } = (0, _helpers2.getMaxMinNumberOfParams)(fnDefinition);
    if (max === min) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentNumber',
        values: {
          fn: fn.name,
          numArgs: max,
          passedArgs: fn.args.length
        },
        locations: fn.location
      }));
    } else if (fn.args.length > max) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentNumberTooMany',
        values: {
          fn: fn.name,
          numArgs: max,
          passedArgs: fn.args.length,
          extraArgs: fn.args.length - max
        },
        locations: fn.location
      }));
    } else {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentNumberTooFew',
        values: {
          fn: fn.name,
          numArgs: min,
          passedArgs: fn.args.length,
          missingArgs: min - fn.args.length
        },
        locations: fn.location
      }));
    }
  }
  // now perform the same check on all functions args
  for (let i = 0; i < fn.args.length; i++) {
    const arg = fn.args[i];
    const allMatchingArgDefinitionsAreConstantOnly = matchingSignatures.every(signature => {
      var _signature$params$i;
      return (_signature$params$i = signature.params[i]) === null || _signature$params$i === void 0 ? void 0 : _signature$params$i.constantOnly;
    });
    const wrappedArray = Array.isArray(arg) ? arg : [arg];
    for (const _subArg of wrappedArray) {
      /**
       * we need to remove the inline casts
       * to see if there's a function under there
       *
       * e.g. for ABS(CEIL(numberField)::int), we need to validate CEIL(numberField)
       */
      const subArg = removeInlineCasts(_subArg);
      if ((0, _helpers.isFunctionItem)(subArg)) {
        const messagesFromArg = validateFunction({
          fn: subArg,
          parentCommand,
          parentOption,
          references,
          /**
           * The constantOnly constraint needs to be enforced for arguments that
           * are functions as well, regardless of whether the definition for the
           * sub function's arguments includes the constantOnly flag.
           *
           * Example:
           * bucket(@timestamp, abs(bytes), "", "")
           *
           * In the above example, the abs function is not defined with the
           * constantOnly flag, but the second parameter in bucket _is_ defined
           * with the constantOnly flag.
           *
           * Because of this, the abs function's arguments inherit the constraint
           * and each should be validated as if each were constantOnly.
           */
          forceConstantOnly: allMatchingArgDefinitionsAreConstantOnly || forceConstantOnly,
          // use the nesting flag for now just for stats and metrics
          // TODO: revisit this part later on to make it more generic
          isNested: ['stats', 'inlinestats', 'metrics'].includes(parentCommand) ? isNested || !(0, _helpers.isAssignment)(fn) : false,
          parentAst
        });
        if (messagesFromArg.some(({
          code
        }) => code === 'expectedConstant')) {
          const consolidatedMessage = (0, _errors.getMessageFromId)({
            messageId: 'expectedConstant',
            values: {
              fn: fn.name,
              given: subArg.text
            },
            locations: subArg.location
          });
          messages.push(consolidatedMessage, ...messagesFromArg.filter(({
            code
          }) => code !== 'expectedConstant'));
        } else {
          messages.push(...messagesFromArg);
        }
      }
    }
  }
  // check if the definition has some specific validation to apply:
  if (fnDefinition.validate) {
    const payloads = fnDefinition.validate(fn);
    if (payloads.length) {
      messages.push(...payloads);
    }
  }
  // at this point we're sure that at least one signature is matching
  const failingSignatures = [];
  let relevantFuncSignatures = matchingSignatures;
  const enrichedArgs = fn.args;
  if (fn.name === 'in' || fn.name === 'not_in') {
    for (let argIndex = 1; argIndex < fn.args.length; argIndex++) {
      relevantFuncSignatures = fnDefinition.signatures.filter(s => {
        var _s$params;
        return ((_s$params = s.params) === null || _s$params === void 0 ? void 0 : _s$params.length) >= argIndex && s.params.slice(0, argIndex).every(({
          type: dataType
        }, idx) => {
          const arg = enrichedArgs[idx];
          if ((0, _helpers.isLiteralItem)(arg)) {
            return dataType === arg.literalType || (0, _esql_types.compareTypesWithLiterals)(dataType, arg.literalType);
          }
          return false; // Non-literal arguments don't match
        });
      });
    }
  }
  for (const signature of relevantFuncSignatures) {
    const failingSignature = [];
    fn.args.forEach((outerArg, index) => {
      const argDef = (0, _helpers.getParamAtPosition)(signature, index);
      if (!outerArg && argDef !== null && argDef !== void 0 && argDef.optional || !argDef) {
        // that's ok, just skip it
        // the else case is already catched with the argument counts check
        // few lines above
        return;
      }

      // check every element of the argument (may be an array of elements, or may be a single element)
      const hasMultipleElements = Array.isArray(outerArg);
      const argElements = hasMultipleElements ? outerArg : [outerArg];
      const singularType = (0, _helpers.extractSingularType)(argDef.type);
      const messagesFromAllArgElements = argElements.flatMap(arg => {
        return [validateFunctionLiteralArg, validateNestedFunctionArg, validateFunctionColumnArg, validateInlineCastArg].flatMap(validateFn => {
          return validateFn(fn, arg, {
            ...argDef,
            type: singularType,
            constantOnly: forceConstantOnly || argDef.constantOnly
          }, references, parentCommand);
        });
      });
      const shouldCollapseMessages = (0, _helpers.isArrayType)(argDef.type) && hasMultipleElements;
      failingSignature.push(...(shouldCollapseMessages ? (0, _helpers2.collapseWrongArgumentTypeMessages)(messagesFromAllArgElements, outerArg, fn.name, argDef.type, parentCommand, references) : messagesFromAllArgElements));
    });
    if (failingSignature.length) {
      failingSignatures.push(failingSignature);
    }
  }
  if (failingSignatures.length && failingSignatures.length === relevantFuncSignatures.length) {
    const failingSignatureOrderedByErrorCount = failingSignatures.map((arr, index) => ({
      index,
      count: arr.length
    })).sort((a, b) => a.count - b.count);
    const indexForShortestFailingsignature = failingSignatureOrderedByErrorCount[0].index;
    messages.push(...failingSignatures[indexForShortestFailingsignature]);
  }
  // This is due to a special case in enrich where an implicit assignment is possible
  // so the AST needs to store an explicit "columnX = columnX" which duplicates the message
  return (0, _uniqBy.default)(messages, ({
    location
  }) => `${location.min}-${location.max}`);
}
function validateSetting(setting, settingDef, command, referenceMaps) {
  const messages = [];
  if (setting.incomplete || command.incomplete) {
    return messages;
  }
  if (!settingDef) {
    const commandDef = (0, _helpers.getCommandDefinition)(command.name);
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unsupportedSetting',
      values: {
        setting: setting.name,
        expected: commandDef.modes.map(({
          name
        }) => name).join(', ')
      },
      locations: setting.location
    }));
    return messages;
  }
  if (settingDef.values.every(({
    name
  }) => name !== setting.name) ||
  // enforce the check on the prefix if present
  settingDef.prefix && !setting.text.startsWith(settingDef.prefix)) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unsupportedSettingCommandValue',
      values: {
        command: command.name.toUpperCase(),
        value: setting.text,
        // for some reason all this enums are uppercase in ES
        expected: settingDef.values.map(({
          name
        }) => `${settingDef.prefix || ''}${name}`).join(', ').toUpperCase()
      },
      locations: setting.location
    }));
  }
  return messages;
}

/**
 * Validate that a function is an aggregate function or that all children
 * recursively terminate at either a literal or an aggregate function.
 */
const isFunctionAggClosed = fn => (0, _helpers.isMaybeAggFunction)(fn) || areFunctionArgsAggClosed(fn);
const areFunctionArgsAggClosed = fn => fn.args.every(arg => (0, _helpers.isLiteralItem)(arg) || (0, _helpers.isFunctionItem)(arg) && isFunctionAggClosed(arg)) || (0, _helpers.isFunctionOperatorParam)(fn);

/**
 * Looks for first nested aggregate function in an aggregate function, recursively.
 */
const findNestedAggFunctionInAggFunction = agg => {
  for (const arg of agg.args) {
    if ((0, _helpers.isFunctionItem)(arg)) {
      return (0, _helpers.isMaybeAggFunction)(arg) ? arg : findNestedAggFunctionInAggFunction(arg);
    }
  }
};

/**
 * Looks for first nested aggregate function in another aggregate a function,
 * recursively.
 *
 * @param fn Function to check for nested aggregate functions.
 * @param parentIsAgg Whether the parent function of `fn` is an aggregate function.
 * @returns The first nested aggregate function in `fn`, or `undefined` if none is found.
 */
const findNestedAggFunction = (fn, parentIsAgg = false) => {
  if ((0, _helpers.isMaybeAggFunction)(fn)) {
    return parentIsAgg ? fn : findNestedAggFunctionInAggFunction(fn);
  }
  for (const arg of fn.args) {
    if ((0, _helpers.isFunctionItem)(arg)) {
      const nestedAgg = findNestedAggFunction(arg, parentIsAgg || (0, _helpers.isAggFunction)(fn));
      if (nestedAgg) return nestedAgg;
    }
  }
};

/**
 * Validates aggregates fields: `... <aggregates> ...`.
 */
const validateAggregates = (command, aggregates, references) => {
  const messages = [];

  // Should never happen.
  if (!aggregates.length) {
    messages.push(_errors.errors.unexpected(command.location));
    return messages;
  }
  let hasMissingAggregationFunctionError = false;
  for (const aggregate of aggregates) {
    if ((0, _helpers.isFunctionItem)(aggregate)) {
      messages.push(...validateFunction({
        fn: aggregate,
        parentCommand: command.name,
        parentOption: undefined,
        references
      }));
      let hasAggregationFunction = false;
      (0, _esqlAst.walk)(aggregate, {
        visitFunction: fn => {
          const definition = (0, _helpers.getFunctionDefinition)(fn.name);
          if (!definition) return;
          if (definition.type === 'agg') hasAggregationFunction = true;
        }
      });
      if (!hasAggregationFunction) {
        hasMissingAggregationFunctionError = true;
        messages.push(_errors.errors.noAggFunction(command, aggregate));
      }
    } else if ((0, _helpers.isColumnItem)(aggregate) || (0, _esqlAst.isIdentifier)(aggregate)) {
      messages.push(_errors.errors.unknownAggFunction(aggregate));
    } else {
      // Should never happen.
    }
  }
  if (hasMissingAggregationFunctionError) {
    return messages;
  }
  for (const aggregate of aggregates) {
    if ((0, _helpers.isFunctionItem)(aggregate)) {
      const fn = (0, _helpers.isAssignment)(aggregate) ? aggregate.args[1] : aggregate;
      if ((0, _helpers.isFunctionItem)(fn) && !isFunctionAggClosed(fn)) {
        messages.push(_errors.errors.expressionNotAggClosed(command, fn));
      }
    }
  }
  if (messages.length) {
    return messages;
  }
  for (const aggregate of aggregates) {
    if ((0, _helpers.isFunctionItem)(aggregate)) {
      const aggInAggFunction = findNestedAggFunction(aggregate);
      if (aggInAggFunction) {
        messages.push(_errors.errors.aggInAggFunction(aggInAggFunction));
        break;
      }
    }
  }
  return messages;
};

/**
 * Validates grouping fields of the BY clause: `... BY <grouping>`.
 */
const validateByGrouping = (fields, commandName, referenceMaps, multipleParams) => {
  const messages = [];
  for (const field of fields) {
    if (!Array.isArray(field)) {
      if (!multipleParams) {
        if ((0, _helpers.isColumnItem)(field)) {
          messages.push(...validateColumnForCommand(field, commandName, referenceMaps));
        }
      } else {
        if ((0, _helpers.isColumnItem)(field)) {
          messages.push(...validateColumnForCommand(field, commandName, referenceMaps));
        }
        if ((0, _helpers.isFunctionItem)(field)) {
          messages.push(...validateFunction({
            fn: field,
            parentCommand: commandName,
            parentOption: 'by',
            references: referenceMaps
          }));
        }
      }
    }
  }
  return messages;
};
function validateOption(option, optionDef, command, referenceMaps) {
  // check if the arguments of the option are of the correct type
  const messages = [];
  if (option.incomplete || command.incomplete) {
    return messages;
  }
  if (!optionDef) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unknownOption',
      values: {
        command: command.name.toUpperCase(),
        option: option.name
      },
      locations: option.location
    }));
    return messages;
  }
  // use dedicate validate fn if provided
  if (optionDef.validate) {
    const fields = _constants.METADATA_FIELDS;
    messages.push(...optionDef.validate(option, command, new Set(fields)));
  }
  if (!optionDef.skipCommonValidation) {
    option.args.forEach(arg => {
      if (!Array.isArray(arg)) {
        if (!optionDef.signature.multipleParams) {
          if ((0, _helpers.isColumnItem)(arg)) {
            messages.push(...validateColumnForCommand(arg, command.name, referenceMaps));
          }
        } else {
          if ((0, _helpers.isColumnItem)(arg)) {
            messages.push(...validateColumnForCommand(arg, command.name, referenceMaps));
          }
          if ((0, _helpers.isFunctionItem)(arg)) {
            messages.push(...validateFunction({
              fn: arg,
              parentCommand: command.name,
              parentOption: option.name,
              references: referenceMaps
            }));
          }
        }
      }
    });
  }
  return messages;
}
function validateSource(source, commandName, {
  sources,
  policies
}) {
  const messages = [];
  if (source.incomplete) {
    return messages;
  }
  const commandDef = (0, _helpers.getCommandDefinition)(commandName);
  const isWildcardAndNotSupported = (0, _helpers.hasWildcard)(source.name) && !commandDef.signature.params.some(({
    wildcards
  }) => wildcards);
  if (isWildcardAndNotSupported) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'wildcardNotSupportedForCommand',
      values: {
        command: commandName.toUpperCase(),
        value: source.name
      },
      locations: source.location
    }));
  } else {
    if (source.sourceType === 'index' && !(0, _helpers.sourceExists)(source.name, sources)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unknownIndex',
        values: {
          name: source.name
        },
        locations: source.location
      }));
    } else if (source.sourceType === 'policy' && !policies.has(source.name)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unknownPolicy',
        values: {
          name: source.name
        },
        locations: source.location
      }));
    }
  }
  return messages;
}
function validateColumnForCommand(column, commandName, references) {
  const messages = [];
  if (commandName === 'row') {
    if (!references.variables.has(column.name) && !(0, _helpers.isParametrized)(column)) {
      messages.push(_errors.errors.unknownColumn(column));
    }
  } else {
    const columnName = (0, _helpers.getQuotedColumnName)(column);
    if ((0, _helpers.getColumnExists)(column, references)) {
      const commandDef = (0, _helpers.getCommandDefinition)(commandName);
      const columnParamsWithInnerTypes = commandDef.signature.params.filter(({
        type,
        innerTypes
      }) => type === 'column' && innerTypes);
      // this should be guaranteed by the columnCheck above
      const columnRef = (0, _helpers.getColumnForASTNode)(column, references);
      if (columnParamsWithInnerTypes.length) {
        const hasSomeWrongInnerTypes = columnParamsWithInnerTypes.every(({
          innerTypes
        }) => innerTypes && !innerTypes.includes('any') && !innerTypes.some(type => (0, _esql_types.compareTypesWithLiterals)(type, columnRef.type)));
        if (hasSomeWrongInnerTypes) {
          const supportedTypes = columnParamsWithInnerTypes.map(({
            innerTypes
          }) => innerTypes).flat().filter(type => type !== undefined);
          messages.push((0, _errors.getMessageFromId)({
            messageId: 'unsupportedColumnTypeForCommand',
            values: {
              command: commandName.toUpperCase(),
              type: supportedTypes.join(', '),
              typeCount: supportedTypes.length,
              givenType: columnRef.type,
              column: columnName
            },
            locations: column.location
          }));
        }
      }
      if ((0, _helpers.hasWildcard)(columnName) && !(0, _helpers.isVariable)(columnRef) && !commandDef.signature.params.some(({
        type,
        wildcards
      }) => type === 'column' && wildcards)) {
        messages.push((0, _errors.getMessageFromId)({
          messageId: 'wildcardNotSupportedForCommand',
          values: {
            command: commandName.toUpperCase(),
            value: columnName
          },
          locations: column.location
        }));
      }
    } else {
      if (column.name) {
        messages.push(_errors.errors.unknownColumn(column));
      }
    }
  }
  return messages;
}
function validateSources(command, sources, references) {
  const messages = [];
  for (const source of sources) {
    messages.push(...validateSource(source, command.name, references));
  }
  return messages;
}

/**
 * Validates the METRICS source command:
 *
 *     METRICS <sources> [ <aggregates> [ BY <grouping> ]]
 */
const validateMetricsCommand = (command, references) => {
  const messages = [];
  const {
    sources,
    aggregates,
    grouping
  } = command;

  // METRICS <sources> ...
  messages.push(...validateSources(command, sources, references));

  // ... <aggregates> ...
  if (aggregates && aggregates.length) {
    messages.push(...validateAggregates(command, aggregates, references));

    // ... BY <grouping>
    if (grouping && grouping.length) {
      messages.push(...validateByGrouping(grouping, 'metrics', references, true));
    }
  }
  return messages;
};

/**
 * Validates the JOIN command:
 *
 *     <LEFT | RIGHT | LOOKUP> JOIN <target> ON <conditions>
 *     <LEFT | RIGHT | LOOKUP> JOIN index [ = alias ] ON <condition> [, <condition> [, ...]]
 */
const validateJoinCommand = (command, references) => {
  const messages = [];
  const {
    commandType,
    args
  } = command;
  const {
    joinIndices
  } = references;
  if (!['left', 'right', 'lookup'].includes(commandType)) {
    return [_errors.errors.unexpected(command.location, 'JOIN command type')];
  }
  const target = args[0];
  let index;
  let alias;
  if ((0, _esqlAst.isBinaryExpression)(target)) {
    if (target.name === 'as') {
      alias = target.args[1];
      index = target.args[0];
      if (!(0, _esqlAst.isSource)(index) || !(0, _esqlAst.isIdentifier)(alias)) {
        return [_errors.errors.unexpected(target.location)];
      }
    } else {
      return [_errors.errors.unexpected(target.location)];
    }
  } else if ((0, _esqlAst.isSource)(target)) {
    index = target;
  } else {
    return [_errors.errors.unexpected(target.location)];
  }
  let isIndexFound = false;
  for (const {
    name,
    aliases
  } of joinIndices) {
    if (index.name === name) {
      isIndexFound = true;
      break;
    }
    if (aliases) {
      for (const aliasName of aliases) {
        if (index.name === aliasName) {
          isIndexFound = true;
          break;
        }
      }
    }
  }
  if (!isIndexFound) {
    const error = _errors.errors.invalidJoinIndex(index);
    messages.push(error);
    return messages;
  }
  return messages;
};
function validateCommand(command, references, ast, currentCommandIndex) {
  const messages = [];
  if (command.incomplete) {
    return messages;
  }
  // do not check the command exists, the grammar is already picking that up
  const commandDef = (0, _helpers.getCommandDefinition)(command.name);
  if (!commandDef) {
    return messages;
  }
  if (commandDef.validate) {
    messages.push(...commandDef.validate(command));
  }
  switch (commandDef.name) {
    case 'metrics':
      {
        const metrics = command;
        messages.push(...validateMetricsCommand(metrics, references));
        break;
      }
    case 'join':
      {
        const join = command;
        const joinCommandErrors = validateJoinCommand(join, references);
        messages.push(...joinCommandErrors);
        break;
      }
    default:
      {
        // Now validate arguments
        for (const commandArg of command.args) {
          const wrappedArg = Array.isArray(commandArg) ? commandArg : [commandArg];
          for (const arg of wrappedArg) {
            if ((0, _helpers.isFunctionItem)(arg)) {
              messages.push(...validateFunction({
                fn: arg,
                parentCommand: command.name,
                parentOption: undefined,
                references,
                parentAst: ast,
                currentCommandIndex
              }));
            } else if ((0, _helpers.isSettingItem)(arg)) {
              messages.push(...validateSetting(arg, commandDef.modes[0], command, references));
            } else if ((0, _helpers.isOptionItem)(arg)) {
              messages.push(...validateOption(arg, commandDef.options.find(({
                name
              }) => name === arg.name), command, references));
            } else if ((0, _helpers.isColumnItem)(arg) || (0, _esqlAst.isIdentifier)(arg)) {
              if (command.name === 'stats' || command.name === 'inlinestats') {
                messages.push(_errors.errors.unknownAggFunction(arg));
              } else {
                messages.push(...validateColumnForCommand(arg, command.name, references));
              }
            } else if ((0, _helpers.isTimeIntervalItem)(arg)) {
              messages.push((0, _errors.getMessageFromId)({
                messageId: 'unsupportedTypeForCommand',
                values: {
                  command: command.name.toUpperCase(),
                  type: 'date_period',
                  value: arg.name
                },
                locations: arg.location
              }));
            } else if ((0, _helpers.isSourceItem)(arg)) {
              messages.push(...validateSource(arg, command.name, references));
            }
          }
        }
      }
  }

  // no need to check for mandatory options passed
  // as they are already validated at syntax level
  return messages;
}
function validateFieldsShadowing(fields, variables) {
  const messages = [];
  for (const variable of variables.keys()) {
    if (fields.has(variable)) {
      var _fields$get;
      const variableHits = variables.get(variable);
      if (!(0, _helpers.areFieldAndVariableTypesCompatible)((_fields$get = fields.get(variable)) === null || _fields$get === void 0 ? void 0 : _fields$get.type, variableHits[0].type)) {
        const fieldType = fields.get(variable).type;
        const variableType = variableHits[0].type;
        const flatFieldType = fieldType;
        const flatVariableType = variableType;
        messages.push((0, _errors.getMessageFromId)({
          messageId: 'shadowFieldType',
          values: {
            field: variable,
            fieldType: flatFieldType,
            newType: flatVariableType
          },
          locations: variableHits[0].location
        }));
      }
    }
  }
  return messages;
}
function validateUnsupportedTypeFields(fields, ast) {
  const usedColumnsInQuery = [];
  (0, _esqlAst.walk)(ast, {
    visitColumn: node => usedColumnsInQuery.push(node.name)
  });
  const messages = [];
  for (const column of usedColumnsInQuery) {
    if (fields.has(column) && fields.get(column).type === 'unsupported') {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unsupportedFieldType',
        values: {
          field: column
        },
        locations: {
          min: 1,
          max: 1
        }
      }));
    }
  }
  return messages;
}
const ignoreErrorsMap = exports.ignoreErrorsMap = {
  getColumnsFor: ['unknownColumn', 'wrongArgumentType', 'unsupportedFieldType'],
  getSources: ['unknownIndex'],
  getPolicies: ['unknownPolicy'],
  getPreferences: [],
  getFieldsMetadata: [],
  getVariablesByType: [],
  canSuggestVariables: [],
  getJoinIndices: []
};

/**
 * ES|QL validation public API
 * It takes a query string and returns a list of messages (errors and warnings) after validate
 * The astProvider is optional, but if not provided the default one from '@kbn/esql-validation-autocomplete' will be used.
 * This is useful for async loading the ES|QL parser and reduce the bundle size, or to swap grammar version.
 * As for the callbacks, while optional, the validation function will selectively ignore some errors types based on each callback missing.
 */
async function validateQuery(queryString, astProvider, options = {}, callbacks) {
  const result = await validateAst(queryString, astProvider, callbacks);
  // early return if we do not want to ignore errors
  if (!options.ignoreOnMissingCallbacks) {
    return result;
  }
  const finalCallbacks = callbacks || {};
  const errorTypoesToIgnore = Object.entries(ignoreErrorsMap).reduce((acc, [key, errorCodes]) => {
    if (!(key in finalCallbacks) || key in finalCallbacks && finalCallbacks[key] == null) {
      for (const e of errorCodes) {
        acc[e] = true;
      }
    }
    return acc;
  }, {});
  const filteredErrors = result.errors.filter(error => {
    if ('severity' in error) {
      return true;
    }
    return !errorTypoesToIgnore[error.code];
  }).map(error => 'severity' in error ? {
    text: error.message,
    code: error.code,
    type: 'error',
    location: {
      min: error.startColumn,
      max: error.endColumn
    }
  } : error);
  return {
    errors: filteredErrors,
    warnings: result.warnings
  };
}

/**
 * This function will perform an high level validation of the
 * query AST. An initial syntax validation is already performed by the parser
 * while here it can detect things like function names, types correctness and potential warnings
 * @param ast A valid AST data structure
 */
async function validateAst(queryString, astProvider, callbacks) {
  var _callbacks$getJoinInd;
  const messages = [];
  const parsingResult = await astProvider(queryString);
  const {
    ast
  } = parsingResult;
  const [sources, availableFields, availablePolicies, joinIndices] = await Promise.all([
  // retrieve the list of available sources
  (0, _resources.retrieveSources)(ast, callbacks),
  // retrieve available fields (if a source command has been defined)
  (0, _resources.retrieveFields)(queryString, ast, callbacks),
  // retrieve available policies (if an enrich command has been defined)
  (0, _resources.retrievePolicies)(ast, callbacks), // retrieve indices for join command
  callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getJoinInd = callbacks.getJoinIndices) === null || _callbacks$getJoinInd === void 0 ? void 0 : _callbacks$getJoinInd.call(callbacks)]);
  if (availablePolicies.size) {
    const fieldsFromPoliciesMap = await (0, _resources.retrievePoliciesFields)(ast, availablePolicies, callbacks);
    fieldsFromPoliciesMap.forEach((value, key) => availableFields.set(key, value));
  }
  if (ast.some(({
    name
  }) => ['grok', 'dissect'].includes(name))) {
    const fieldsFromGrokOrDissect = await (0, _resources.retrieveFieldsFromStringSources)(queryString, ast, callbacks);
    fieldsFromGrokOrDissect.forEach((value, key) => {
      // if the field is already present, do not overwrite it
      // Note: this can also overlap with some variables
      if (!availableFields.has(key)) {
        availableFields.set(key, value);
      }
    });
  }
  const variables = (0, _variables.collectVariables)(ast, availableFields, queryString);
  // notify if the user is rewriting a column as variable with another type
  messages.push(...validateFieldsShadowing(availableFields, variables));
  messages.push(...validateUnsupportedTypeFields(availableFields, ast));
  for (const [index, command] of ast.entries()) {
    const references = {
      sources,
      fields: availableFields,
      policies: availablePolicies,
      variables,
      query: queryString,
      joinIndices: (joinIndices === null || joinIndices === void 0 ? void 0 : joinIndices.indices) || []
    };
    const commandMessages = validateCommand(command, references, ast, index);
    messages.push(...commandMessages);
  }
  return {
    errors: [...parsingResult.errors, ...messages.filter(({
      type
    }) => type === 'error')],
    warnings: messages.filter(({
      type
    }) => type === 'warning')
  };
}