"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isWhereExpression = exports.isSource = exports.isProperNode = exports.isParamLiteral = exports.isLiteral = exports.isIntegerLiteral = exports.isIdentifier = exports.isFunctionExpression = exports.isFieldExpression = exports.isDoubleLiteral = exports.isColumn = exports.isBinaryExpression = exports.isAsExpression = exports.binaryExpressionGroup = void 0;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const isProperNode = node => !!node && typeof node === 'object' && !Array.isArray(node);
exports.isProperNode = isProperNode;
const isFunctionExpression = node => isProperNode(node) && node.type === 'function';

/**
 * Returns true if the given node is a binary expression, i.e. an operator
 * surrounded by two operands:
 *
 * ```
 * 1 + 1
 * column LIKE "foo"
 * foo = "bar"
 * ```
 *
 * @param node Any ES|QL AST node.
 */
exports.isFunctionExpression = isFunctionExpression;
const isBinaryExpression = node => isFunctionExpression(node) && node.subtype === 'binary-expression';
exports.isBinaryExpression = isBinaryExpression;
const isWhereExpression = node => isBinaryExpression(node) && node.name === 'where';
exports.isWhereExpression = isWhereExpression;
const isAsExpression = node => isBinaryExpression(node) && node.name === 'as';
exports.isAsExpression = isAsExpression;
const isFieldExpression = node => isBinaryExpression(node) && node.name === '=';
exports.isFieldExpression = isFieldExpression;
const isLiteral = node => isProperNode(node) && node.type === 'literal';
exports.isLiteral = isLiteral;
const isIntegerLiteral = node => isLiteral(node) && node.literalType === 'integer';
exports.isIntegerLiteral = isIntegerLiteral;
const isDoubleLiteral = node => isLiteral(node) && node.literalType === 'double';
exports.isDoubleLiteral = isDoubleLiteral;
const isParamLiteral = node => isLiteral(node) && node.literalType === 'param';
exports.isParamLiteral = isParamLiteral;
const isColumn = node => isProperNode(node) && node.type === 'column';
exports.isColumn = isColumn;
const isSource = node => isProperNode(node) && node.type === 'source';
exports.isSource = isSource;
const isIdentifier = node => isProperNode(node) && node.type === 'identifier';

/**
 * Returns the group of a binary expression:
 *
 * - `additive`: `+`, `-`
 * - `multiplicative`: `*`, `/`, `%`
 * - `assignment`: `=`
 * - `comparison`: `==`, `=~`, `!=`, `<`, `<=`, `>`, `>=`
 * - `regex`: `like`, `not_like`, `rlike`, `not_rlike`
 * @param node Any ES|QL AST node.
 * @returns Binary expression group or undefined if the node is not a binary expression.
 */
exports.isIdentifier = isIdentifier;
const binaryExpressionGroup = node => {
  if (isBinaryExpression(node)) {
    switch (node.name) {
      case '+':
      case '-':
        return _constants.BinaryExpressionGroup.additive;
      case '*':
      case '/':
      case '%':
        return _constants.BinaryExpressionGroup.multiplicative;
      case '=':
        return _constants.BinaryExpressionGroup.assignment;
      case '==':
      case '=~':
      case '!=':
      case '<':
      case '<=':
      case '>':
      case '>=':
        return _constants.BinaryExpressionGroup.comparison;
      case 'like':
      case 'not_like':
      case 'rlike':
      case 'not_rlike':
        return _constants.BinaryExpressionGroup.regex;
    }
  }
  return _constants.BinaryExpressionGroup.unknown;
};
exports.binaryExpressionGroup = binaryExpressionGroup;