"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sourceDefinitionRoutes = void 0;
var _zod = require("@kbn/zod");
var _constants = require("../../lib/v2/constants");
var _types = require("../../lib/v2/types");
var _create_entity_manager_server_route = require("../create_entity_manager_server_route");
var _entity_definition_conflict = require("../../lib/v2/errors/entity_definition_conflict");
var _unknown_entity_type = require("../../lib/v2/errors/unknown_entity_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createSourceDefinitionRoute = (0, _create_entity_manager_server_route.createEntityManagerServerRoute)({
  endpoint: 'POST /internal/entities/v2/definitions/sources',
  security: {
    authz: {
      requiredPrivileges: [_constants.CREATE_ENTITY_SOURCE_DEFINITION_PRIVILEGE]
    }
  },
  params: _zod.z.object({
    body: _zod.z.object({
      source: _types.entitySourceDefinitionRt
    })
  }),
  handler: async ({
    request,
    response,
    params,
    getScopedClient
  }) => {
    try {
      const client = await getScopedClient({
        request
      });
      const source = await client.v2.storeSourceDefinition(params.body.source);
      return response.created({
        body: {
          source
        },
        headers: {
          location: `GET /internal/entities/v2/definitions/sources/${source.id}`
        }
      });
    } catch (error) {
      if (error instanceof _unknown_entity_type.UnknownEntityType) {
        return response.notFound({
          body: {
            message: error.message
          }
        });
      }
      if (error instanceof _entity_definition_conflict.EntityDefinitionConflict) {
        return response.conflict({
          body: {
            message: error.message
          }
        });
      }
      throw error;
    }
  }
});
const readSourceDefinitionsRoute = (0, _create_entity_manager_server_route.createEntityManagerServerRoute)({
  endpoint: 'GET /internal/entities/v2/definitions/sources',
  security: {
    authz: {
      requiredPrivileges: [_constants.READ_ENTITY_SOURCE_DEFINITION_PRIVILEGE]
    }
  },
  handler: async ({
    request,
    response,
    getScopedClient
  }) => {
    const client = await getScopedClient({
      request
    });
    const sources = await client.v2.readSourceDefinitions();
    return response.ok({
      body: {
        sources
      }
    });
  }
});
const sourceDefinitionRoutes = exports.sourceDefinitionRoutes = {
  ...createSourceDefinitionRoute,
  ...readSourceDefinitionsRoute
};