"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.asKeyword = asKeyword;
exports.defaultSort = defaultSort;
exports.isRejectedResult = exports.isFulfilledResult = void 0;
exports.mergeEntitiesList = mergeEntitiesList;
exports.sortEntitiesList = sortEntitiesList;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getLatestDate(date1, date2) {
  if (!date1 && !date2) return;
  return new Date(Math.max(date1 ? Date.parse(date1) : 0, date2 ? Date.parse(date2) : 0)).toISOString();
}
function mergeEntities(identityFields, mergeableFields, entity1, entity2) {
  const merged = {
    ...entity1
  };
  const latestTimestamp = getLatestDate(entity1['entity.last_seen_timestamp'], entity2['entity.last_seen_timestamp']);
  if (latestTimestamp) {
    merged['entity.last_seen_timestamp'] = latestTimestamp;
  }
  for (const [key, value] of Object.entries(entity2).filter(([_key]) => mergeableFields.includes(_key) && entity2[_key])) {
    if (merged[key]) {
      // we want to keep identity fields as single-value properties.
      // this can happen if two sources group by the same identity
      if (!identityFields.includes(key)) {
        merged[key] = (0, _lodash.uniq)([...(Array.isArray(merged[key]) ? merged[key] : [merged[key]]), ...(Array.isArray(value) ? value : [value])]);
      }
    } else {
      merged[key] = value;
    }
  }
  return merged;
}
function mergeEntitiesList({
  entities,
  sources,
  metadataFields
}) {
  const identityFields = (0, _lodash.uniq)([...sources.flatMap(source => source.identity_fields)]);
  const mergeableFields = (0, _lodash.uniq)([...identityFields, ...metadataFields, ...sources.flatMap(source => source.metadata_fields)]);
  const instances = {};
  for (let i = 0; i < entities.length; i++) {
    const entity = entities[i];
    const id = entity['entity.id'];
    if (instances[id]) {
      instances[id] = mergeEntities(identityFields, mergeableFields, instances[id], entity);
    } else {
      instances[id] = entity;
    }
  }
  return Object.values(instances);
}
function sortEntitiesList({
  entities,
  sources,
  sort
}) {
  if (!sort) {
    sort = defaultSort(sources);
  }
  return (0, _lodash.orderBy)(entities, sort.field, sort.direction.toLowerCase());
}
function asKeyword(field) {
  return `${field}::keyword`;
}
function defaultSort(sources) {
  if (sources.some(source => source.timestamp_field)) {
    return {
      field: 'entity.last_seen_timestamp',
      direction: 'DESC'
    };
  }
  return {
    field: 'entity.id',
    direction: 'ASC'
  };
}
const isRejectedResult = input => input.status === 'rejected';
exports.isRejectedResult = isRejectedResult;
const isFulfilledResult = input => input.status === 'fulfilled';
exports.isFulfilledResult = isFulfilledResult;