"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EntityClient = void 0;
var _source_definition = require("./definitions/source_definition");
var _type_definition = require("./definitions/type_definition");
var _queries = require("./queries");
var _utils = require("./queries/utils");
var _unknown_entity_type = require("./errors/unknown_entity_type");
var _run_esql_query = require("./run_esql_query");
var _validate_fields = require("./validate_fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EntityClient {
  constructor(options) {
    this.options = options;
  }
  async searchEntities({
    type,
    ...options
  }) {
    const sources = await this.readSourceDefinitions({
      type
    });
    if (sources.length === 0) {
      throw new _unknown_entity_type.UnknownEntityType(`No sources found for entity type [${type}]`);
    }
    return this.searchEntitiesBySources({
      sources,
      ...options
    });
  }
  async searchEntitiesBySources({
    sources,
    metadata_fields: metadataFields,
    filters,
    start,
    end,
    sort,
    limit
  }) {
    const searches = sources.map(async source => {
      const availableMetadataFields = await (0, _validate_fields.validateFields)({
        source,
        metadataFields,
        esClient: this.options.clusterClient.asCurrentUser,
        logger: this.options.logger
      });
      const {
        query,
        filter
      } = (0, _queries.getEntityInstancesQuery)({
        source: {
          ...source,
          metadata_fields: availableMetadataFields,
          filters: [...source.filters, ...filters]
        },
        start,
        end,
        sort,
        limit
      });
      this.options.logger.debug(() => `Entity instances query: ${query}\nfilter: ${JSON.stringify(filter, null, 2)}`);
      const rawEntities = await (0, _run_esql_query.runESQLQuery)('resolve entities', {
        query,
        filter,
        esClient: this.options.clusterClient.asCurrentUser,
        logger: this.options.logger
      });
      return rawEntities;
    });
    const {
      entities,
      errors
    } = await Promise.allSettled(searches).then(results => ({
      entities: results.filter(_utils.isFulfilledResult).flatMap(result => result.value),
      errors: results.filter(_utils.isRejectedResult).map(result => result.reason.message)
    }));
    if (sources.length === 1) {
      return {
        entities,
        errors
      };
    }

    // we have to manually merge, sort and limit entities since we run
    // independant queries for each source
    return {
      errors,
      entities: (0, _utils.sortEntitiesList)({
        sources,
        sort,
        entities: (0, _utils.mergeEntitiesList)({
          entities,
          sources,
          metadataFields
        })
      }).slice(0, limit)
    };
  }
  async countEntities({
    start,
    end,
    types = [],
    filters = []
  }) {
    if (types.length === 0) {
      types = (await this.readTypeDefinitions()).map(definition => definition.id);
    }
    const counts = await Promise.all(types.map(async type => {
      const sources = await this.readSourceDefinitions({
        type
      });
      if (sources.length === 0) {
        return {
          type,
          value: 0,
          errors: []
        };
      }
      const {
        sources: validSources,
        errors
      } = await Promise.allSettled(sources.map(source => (0, _validate_fields.validateFields)({
        source,
        esClient: this.options.clusterClient.asCurrentUser,
        logger: this.options.logger
      }).then(() => source))).then(results => ({
        sources: results.filter(_utils.isFulfilledResult).flatMap(result => result.value),
        errors: results.filter(_utils.isRejectedResult).map(result => result.reason.message)
      }));
      if (validSources.length === 0) {
        return {
          type,
          value: 0,
          errors
        };
      }
      const {
        query,
        filter
      } = (0, _queries.getEntityCountQuery)({
        sources: validSources,
        filters,
        start,
        end
      });
      this.options.logger.info(`Entity count query: ${query}\nfilter: ${JSON.stringify(filter, null, 2)}`);
      const [{
        count
      }] = await (0, _run_esql_query.runESQLQuery)('count entities', {
        query,
        filter,
        esClient: this.options.clusterClient.asCurrentUser,
        logger: this.options.logger
      });
      return {
        type,
        value: count,
        errors
      };
    }));
    return counts.reduce((result, count) => {
      result.types[count.type] = count.value;
      result.total += count.value;
      result.errors.push(...count.errors);
      return result;
    }, {
      total: 0,
      types: {},
      errors: []
    });
  }
  async storeTypeDefinition(type) {
    return (0, _type_definition.storeTypeDefinition)({
      type,
      clusterClient: this.options.clusterClient,
      logger: this.options.logger
    });
  }
  async readTypeDefinitions() {
    return (0, _type_definition.readTypeDefinitions)(this.options.clusterClient, this.options.logger);
  }
  async storeSourceDefinition(source) {
    return (0, _source_definition.storeSourceDefinition)({
      source,
      clusterClient: this.options.clusterClient,
      logger: this.options.logger
    });
  }
  async readSourceDefinitions(options) {
    return (0, _source_definition.readSourceDefinitions)(this.options.clusterClient, this.options.logger, options);
  }
}
exports.EntityClient = EntityClient;