"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CachedFetchConnectorByIdApiLogic = void 0;
var _kea = require("kea");
var _lodash = require("lodash");
var _api = require("../../../../../common/types/api");
var _fetch_connector_by_id_logic = require("./fetch_connector_by_id_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FETCH_CONNECTOR_POLLING_DURATION = 5000; // 5 seconds
const FETCH_CONNECTOR_POLLING_DURATION_ON_FAILURE = 30000; // 30 seconds

const CachedFetchConnectorByIdApiLogic = exports.CachedFetchConnectorByIdApiLogic = (0, _kea.kea)({
  actions: {
    clearPollTimeout: true,
    createPollTimeout: duration => ({
      duration
    }),
    setTimeoutId: id => ({
      id
    }),
    startPolling: connectorId => ({
      connectorId
    }),
    stopPolling: true
  },
  connect: {
    actions: [_fetch_connector_by_id_logic.FetchConnectorByIdApiLogic, ['apiSuccess', 'apiError', 'apiReset', 'makeRequest']],
    values: [_fetch_connector_by_id_logic.FetchConnectorByIdApiLogic, ['data as fetchConnectorByIdApiData', 'status']]
  },
  events: ({
    values
  }) => ({
    beforeUnmount: () => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
    }
  }),
  listeners: ({
    actions,
    values
  }) => ({
    apiError: () => {
      if (values.pollTimeoutId) {
        actions.createPollTimeout(FETCH_CONNECTOR_POLLING_DURATION_ON_FAILURE);
      }
    },
    apiSuccess: () => {
      if (values.pollTimeoutId) {
        actions.createPollTimeout(FETCH_CONNECTOR_POLLING_DURATION);
      }
    },
    createPollTimeout: ({
      duration
    }) => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
      const timeoutId = setTimeout(() => {
        actions.makeRequest({
          connectorId: values.connectorId
        });
      }, duration);
      actions.setTimeoutId(timeoutId);
    },
    startPolling: ({
      connectorId
    }) => {
      // Recurring polls are created by apiSuccess and apiError, depending on pollTimeoutId
      if (values.pollTimeoutId) {
        if (connectorId === values.connectorId) return;
        clearTimeout(values.pollTimeoutId);
      }
      actions.makeRequest({
        connectorId
      });
      actions.createPollTimeout(FETCH_CONNECTOR_POLLING_DURATION);
    },
    stopPolling: () => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
      actions.clearPollTimeout();
    }
  }),
  path: ['enterprise_search', 'content', 'api', 'fetch_connector_by_id_api_wrapper'],
  reducers: {
    connectorData: [null, {
      apiReset: () => null,
      // @ts-expect-error upgrade typescript v5.1.6
      apiSuccess: (currentState, newConnectorData) => {
        var _newConnectorData$con;
        return (0, _lodash.isEqual)(currentState, newConnectorData.connector) ? currentState : (_newConnectorData$con = newConnectorData.connector) !== null && _newConnectorData$con !== void 0 ? _newConnectorData$con : null;
      }
    }],
    connectorId: ['', {
      apiReset: () => '',
      // @ts-expect-error upgrade typescript v5.1.6
      startPolling: (_, {
        connectorId
      }) => connectorId
    }],
    pollTimeoutId: [null, {
      clearPollTimeout: () => null,
      // @ts-expect-error upgrade typescript v5.1.6
      setTimeoutId: (_, {
        id
      }) => id
    }]
  },
  selectors: ({
    selectors
  }) => ({
    isInitialLoading: [() => [selectors.status, selectors.connectorData], (status, connectorData) => {
      return status === _api.Status.IDLE || connectorData === null && status === _api.Status.LOADING;
    }]
  })
});