"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SourceEnginesLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _recursively_fetch_engines = require("../../utils/recursively_fetch_engines");
var _engine = require("../engine");
var _i18n = require("./i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SourceEnginesLogic = exports.SourceEnginesLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'source_engines_logic'],
  actions: () => ({
    addSourceEngines: sourceEngineNames => ({
      sourceEngineNames
    }),
    fetchIndexedEngines: true,
    fetchSourceEngines: true,
    onSourceEngineRemove: sourceEngineNameToRemove => ({
      sourceEngineNameToRemove
    }),
    onSourceEnginesAdd: sourceEnginesToAdd => ({
      sourceEnginesToAdd
    }),
    onSourceEnginesFetch: sourceEngines => ({
      sourceEngines
    }),
    removeSourceEngine: sourceEngineName => ({
      sourceEngineName
    }),
    setIndexedEngines: indexedEngines => ({
      indexedEngines
    }),
    openModal: true,
    closeModal: true,
    onAddEnginesSelection: selectedEngineNamesToAdd => ({
      selectedEngineNamesToAdd
    })
  }),
  reducers: () => ({
    dataLoading: [true, {
      onSourceEnginesFetch: () => false
    }],
    modalLoading: [false, {
      addSourceEngines: () => true,
      closeModal: () => false
    }],
    isModalOpen: [false, {
      openModal: () => true,
      closeModal: () => false
    }],
    indexedEngines: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setIndexedEngines: (_, {
        indexedEngines
      }) => indexedEngines
    }],
    selectedEngineNamesToAdd: [[], {
      closeModal: () => [],
      // @ts-expect-error upgrade typescript v5.1.6
      onAddEnginesSelection: (_, {
        selectedEngineNamesToAdd
      }) => selectedEngineNamesToAdd
    }],
    sourceEngines: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      onSourceEnginesAdd: (sourceEngines, {
        sourceEnginesToAdd
      }) => [...sourceEngines, ...sourceEnginesToAdd],
      // @ts-expect-error upgrade typescript v5.1.6
      onSourceEnginesFetch: (_, {
        sourceEngines
      }) => sourceEngines,
      // @ts-expect-error upgrade typescript v5.1.6
      onSourceEngineRemove: (sourceEngines, {
        sourceEngineNameToRemove
      }) =>
      // @ts-expect-error upgrade typescript v5.1.6
      sourceEngines.filter(sourceEngine => sourceEngine.name !== sourceEngineNameToRemove)
    }]
  }),
  selectors: {
    indexedEngineNames: [selectors => [selectors.indexedEngines], indexedEngines => indexedEngines.map(engine => engine.name)],
    sourceEngineNames: [selectors => [selectors.sourceEngines], sourceEngines => sourceEngines.map(engine => engine.name)],
    selectableEngineNames: [selectors => [selectors.indexedEngineNames, selectors.sourceEngineNames], (indexedEngineNames, sourceEngineNames) => indexedEngineNames.filter(engineName => !sourceEngineNames.includes(engineName))]
  },
  listeners: ({
    actions,
    values
  }) => ({
    addSourceEngines: async ({
      sourceEngineNames
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        await http.post(`/internal/app_search/engines/${engineName}/source_engines/bulk_create`, {
          body: JSON.stringify({
            source_engine_slugs: sourceEngineNames
          })
        });
        const sourceEnginesToAdd = values.indexedEngines.filter(({
          name
        }) => sourceEngineNames.includes(name));
        actions.onSourceEnginesAdd(sourceEnginesToAdd);
        (0, _flash_messages.flashSuccessToast)((0, _i18n.ADD_SOURCE_ENGINES_SUCCESS_MESSAGE)(sourceEngineNames));
        _engine.EngineLogic.actions.initializeEngine();
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      } finally {
        actions.closeModal();
      }
    },
    fetchSourceEngines: () => {
      const {
        engineName
      } = _engine.EngineLogic.values;
      (0, _recursively_fetch_engines.recursivelyFetchEngines)({
        endpoint: `/internal/app_search/engines/${engineName}/source_engines`,
        onComplete: engines => actions.onSourceEnginesFetch(engines)
      });
    },
    fetchIndexedEngines: () => {
      (0, _recursively_fetch_engines.recursivelyFetchEngines)({
        endpoint: '/internal/app_search/engines',
        onComplete: engines => actions.setIndexedEngines(engines),
        query: {
          type: 'indexed'
        }
      });
    },
    removeSourceEngine: async ({
      sourceEngineName
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        await http.delete(`/internal/app_search/engines/${engineName}/source_engines/${sourceEngineName}`);
        actions.onSourceEngineRemove(sourceEngineName);
        (0, _flash_messages.flashSuccessToast)((0, _i18n.REMOVE_SOURCE_ENGINE_SUCCESS_MESSAGE)(sourceEngineName));

        // Changing source engines can change schema conflicts and invalid boosts,
        // so we re-initialize the engine to re-fetch that data
        _engine.EngineLogic.actions.initializeEngine(); //
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    }
  })
});