"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RoleMappingsLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _has_scoped_engines = require("../../utils/role/has_scoped_engines");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFirstAttributeName = roleMapping => Object.entries(roleMapping.rules)[0][0];
const getFirstAttributeValue = roleMapping => Object.entries(roleMapping.rules)[0][1];
const emptyUser = {
  username: '',
  email: ''
};
const RoleMappingsLogic = exports.RoleMappingsLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'users_and_roles'],
  actions: {
    setRoleMappingsData: data => data,
    setRoleMapping: roleMapping => ({
      roleMapping
    }),
    setElasticsearchUser: elasticsearchUser => ({
      elasticsearchUser
    }),
    setSingleUserRoleMapping: singleUserRoleMapping => ({
      singleUserRoleMapping
    }),
    setRoleMappings: ({
      roleMappings
    }) => ({
      roleMappings
    }),
    setRoleMappingErrors: errors => ({
      errors
    }),
    handleRoleChange: roleType => ({
      roleType
    }),
    handleUsernameSelectChange: username => ({
      username
    }),
    handleEngineSelectionChange: engineNames => ({
      engineNames
    }),
    handleAttributeSelectorChange: (value, firstElasticsearchRole) => ({
      value,
      firstElasticsearchRole
    }),
    handleAttributeValueChange: value => ({
      value
    }),
    handleAccessAllEnginesChange: selected => ({
      selected
    }),
    enableRoleBasedAccess: true,
    openSingleUserRoleMappingFlyout: true,
    setUserExistingRadioValue: userFormUserIsExisting => ({
      userFormUserIsExisting
    }),
    resetState: true,
    initializeRoleMappings: true,
    initializeSingleUserRoleMapping: roleMappingId => ({
      roleMappingId
    }),
    initializeRoleMapping: roleMappingId => ({
      roleMappingId
    }),
    handleDeleteMapping: roleMappingId => ({
      roleMappingId
    }),
    handleSaveMapping: true,
    handleSaveUser: true,
    openRoleMappingFlyout: true,
    closeUsersAndRolesFlyout: false,
    setElasticsearchUsernameValue: username => ({
      username
    }),
    setElasticsearchEmailValue: email => ({
      email
    }),
    setUserCreated: true,
    setUserFormIsNewUser: userFormIsNewUser => ({
      userFormIsNewUser
    })
  },
  reducers: {
    dataLoading: [true, {
      setRoleMappingsData: () => false,
      setRoleMappings: () => false,
      resetState: () => true,
      enableRoleBasedAccess: () => true
    }],
    roleMappings: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingsData: (_, {
        roleMappings
      }) => roleMappings,
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappings: (_, {
        roleMappings
      }) => roleMappings,
      resetState: () => []
    }],
    singleUserRoleMappings: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingsData: (_, {
        singleUserRoleMappings
      }) => singleUserRoleMappings,
      resetState: () => []
    }],
    hasAdvancedRoles: [false, {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingsData: (_, {
        hasAdvancedRoles
      }) => hasAdvancedRoles
    }],
    availableEngines: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingsData: (_, {
        availableEngines
      }) => availableEngines,
      resetState: () => []
    }],
    attributes: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingsData: (_, {
        attributes
      }) => attributes,
      resetState: () => []
    }],
    elasticsearchRoles: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingsData: (_, {
        elasticsearchRoles
      }) => elasticsearchRoles
    }],
    elasticsearchUsers: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingsData: (_, {
        elasticsearchUsers
      }) => elasticsearchUsers,
      resetState: () => []
    }],
    roleMapping: [null, {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMapping: (_, {
        roleMapping
      }) => roleMapping,
      initializeRoleMappings: () => null,
      resetState: () => null,
      closeUsersAndRolesFlyout: () => null
    }],
    roleType: ['owner', {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMapping: (_, {
        roleMapping
      }) => roleMapping.roleType,
      // @ts-expect-error upgrade typescript v5.1.6
      handleRoleChange: (_, {
        roleType
      }) => roleType
    }],
    accessAllEngines: [true, {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMapping: (_, {
        roleMapping
      }) => roleMapping.accessAllEngines,
      // @ts-expect-error upgrade typescript v5.1.6
      handleRoleChange: (_, {
        roleType
      }) => !(0, _has_scoped_engines.roleHasScopedEngines)(roleType),
      // @ts-expect-error upgrade typescript v5.1.6
      handleAccessAllEnginesChange: (_, {
        selected
      }) => selected,
      closeUsersAndRolesFlyout: () => true
    }],
    attributeValue: ['', {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMapping: (_, {
        roleMapping
      }) => getFirstAttributeValue(roleMapping),
      // @ts-expect-error upgrade typescript v5.1.6
      handleAttributeSelectorChange: (_, {
        value,
        firstElasticsearchRole
      }) => value === 'role' ? firstElasticsearchRole : '',
      // @ts-expect-error upgrade typescript v5.1.6
      handleAttributeValueChange: (_, {
        value
      }) => value,
      resetState: () => '',
      closeUsersAndRolesFlyout: () => ''
    }],
    attributeName: ['username', {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMapping: (_, {
        roleMapping
      }) => getFirstAttributeName(roleMapping),
      // @ts-expect-error upgrade typescript v5.1.6
      handleAttributeSelectorChange: (_, {
        value
      }) => value,
      // @ts-expect-error upgrade typescript v5.1.6
      resetState: () => 'username',
      // @ts-expect-error upgrade typescript v5.1.6
      closeUsersAndRolesFlyout: () => 'username'
    }],
    selectedEngines: [new Set(), {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMapping: (_, {
        roleMapping
      }) => new Set(roleMapping.engines.map(engine => engine.name)),
      // @ts-expect-error upgrade typescript v5.1.6
      handleAccessAllEnginesChange: () => new Set(),
      // @ts-expect-error upgrade typescript v5.1.6
      handleEngineSelectionChange: (_, {
        engineNames
      }) => {
        const newSelectedEngineNames = new Set();
        // @ts-expect-error upgrade typescript v5.1.6
        engineNames.forEach(engineName => newSelectedEngineNames.add(engineName));
        return newSelectedEngineNames;
      },
      // @ts-expect-error upgrade typescript v5.1.6
      closeUsersAndRolesFlyout: () => new Set()
    }],
    roleMappingFlyoutOpen: [false, {
      openRoleMappingFlyout: () => true,
      closeUsersAndRolesFlyout: () => false,
      initializeRoleMappings: () => false,
      initializeRoleMapping: () => true
    }],
    singleUserRoleMappingFlyoutOpen: [false, {
      openSingleUserRoleMappingFlyout: () => true,
      closeUsersAndRolesFlyout: () => false,
      initializeSingleUserRoleMapping: () => true
    }],
    singleUserRoleMapping: [null, {
      // @ts-expect-error upgrade typescript v5.1.6
      setSingleUserRoleMapping: (_, {
        singleUserRoleMapping
      }) => singleUserRoleMapping || null,
      closeUsersAndRolesFlyout: () => null
    }],
    roleMappingErrors: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingErrors: (_, {
        errors
      }) => errors,
      handleSaveMapping: () => [],
      closeUsersAndRolesFlyout: () => []
    }],
    userFormUserIsExisting: [true, {
      // @ts-expect-error upgrade typescript v5.1.6
      setUserExistingRadioValue: (_, {
        userFormUserIsExisting
      }) => userFormUserIsExisting,
      closeUsersAndRolesFlyout: () => true
    }],
    elasticsearchUser: [emptyUser, {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingsData: (_, {
        elasticsearchUsers
      }) => elasticsearchUsers[0] || emptyUser,
      // @ts-expect-error upgrade typescript v5.1.6
      setElasticsearchUser: (_, {
        elasticsearchUser
      }) => elasticsearchUser || emptyUser,
      // @ts-expect-error upgrade typescript v5.1.6
      setElasticsearchUsernameValue: (state, {
        username
      }) => ({
        ...state,
        username
      }),
      // @ts-expect-error upgrade typescript v5.1.6
      setElasticsearchEmailValue: (state, {
        email
      }) => ({
        ...state,
        email
      }),
      closeUsersAndRolesFlyout: () => emptyUser
    }],
    userCreated: [false, {
      setUserCreated: () => true,
      closeUsersAndRolesFlyout: () => false
    }],
    userFormIsNewUser: [true, {
      // @ts-expect-error upgrade typescript v5.1.6
      setUserFormIsNewUser: (_, {
        userFormIsNewUser
      }) => userFormIsNewUser
    }],
    smtpSettingsPresent: [false, {
      // @ts-expect-error upgrade typescript v5.1.6
      setRoleMappingsData: (_, {
        smtpSettingsPresent
      }) => smtpSettingsPresent
    }],
    formLoading: [false, {
      handleSaveMapping: () => true,
      handleSaveUser: () => true,
      initializeRoleMappings: () => false,
      setRoleMappingErrors: () => false
    }]
  },
  selectors: ({
    selectors
  }) => ({
    selectedOptions: [() => [selectors.selectedEngines, selectors.availableEngines], (selectedEngines, availableEngines) => {
      const selectedNames = Array.from(selectedEngines.values());
      return availableEngines.filter(({
        name
      }) => selectedNames.includes(name)).map(({
        name
      }) => ({
        label: name,
        value: name
      }));
    }]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    enableRoleBasedAccess: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const route = '/internal/app_search/role_mappings/enable_role_based_access';
      try {
        await http.post(route);
        actions.initializeRoleMappings();
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    initializeRoleMappings: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const route = '/internal/app_search/role_mappings';
      try {
        const response = await http.get(route);
        actions.setRoleMappingsData(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    initializeRoleMapping: async ({
      roleMappingId
    }) => {
      const roleMapping = values.roleMappings.find(({
        id
      }) => id === roleMappingId);
      if (roleMapping) actions.setRoleMapping(roleMapping);
    },
    initializeSingleUserRoleMapping: ({
      roleMappingId
    }) => {
      const singleUserRoleMapping = values.singleUserRoleMappings.find(({
        roleMapping
      }) => roleMapping.id === roleMappingId);
      if (singleUserRoleMapping) {
        actions.setElasticsearchUser(singleUserRoleMapping.elasticsearchUser);
        actions.setRoleMapping(singleUserRoleMapping.roleMapping);
      }
      actions.setSingleUserRoleMapping(singleUserRoleMapping);
      actions.setUserFormIsNewUser(!singleUserRoleMapping);
    },
    handleDeleteMapping: async ({
      roleMappingId
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const route = `/internal/app_search/role_mappings/${roleMappingId}`;
      try {
        await http.delete(route);
        actions.initializeRoleMappings();
        (0, _flash_messages.flashSuccessToast)(_constants.ROLE_MAPPING_DELETED_MESSAGE);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    handleSaveMapping: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        attributeName,
        attributeValue,
        roleType,
        roleMapping,
        accessAllEngines,
        selectedEngines
      } = values;
      const body = JSON.stringify({
        roleType,
        accessAllEngines,
        rules: {
          [attributeName]: attributeValue
        },
        engines: accessAllEngines ? [] : Array.from(selectedEngines)
      });
      const request = !roleMapping ? http.post('/internal/app_search/role_mappings', {
        body
      }) : http.put(`/internal/app_search/role_mappings/${roleMapping.id}`, {
        body
      });
      const SUCCESS_MESSAGE = !roleMapping ? _constants.ROLE_MAPPING_CREATED_MESSAGE : _constants.ROLE_MAPPING_UPDATED_MESSAGE;
      try {
        await request;
        actions.initializeRoleMappings();
        (0, _flash_messages.flashSuccessToast)(SUCCESS_MESSAGE);
      } catch (e) {
        var _e$body, _e$body$attributes;
        actions.setRoleMappingErrors(e === null || e === void 0 ? void 0 : (_e$body = e.body) === null || _e$body === void 0 ? void 0 : (_e$body$attributes = _e$body.attributes) === null || _e$body$attributes === void 0 ? void 0 : _e$body$attributes.errors);
      }
    },
    resetState: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    handleSaveUser: async () => {
      var _singleUserRoleMappin;
      const {
        http
      } = _http.HttpLogic.values;
      const {
        roleType,
        singleUserRoleMapping,
        accessAllEngines,
        selectedEngines,
        elasticsearchUser: {
          email,
          username
        }
      } = values;
      const body = JSON.stringify({
        roleMapping: {
          engines: accessAllEngines ? [] : Array.from(selectedEngines),
          roleType,
          accessAllEngines,
          id: singleUserRoleMapping === null || singleUserRoleMapping === void 0 ? void 0 : (_singleUserRoleMappin = singleUserRoleMapping.roleMapping) === null || _singleUserRoleMappin === void 0 ? void 0 : _singleUserRoleMappin.id
        },
        elasticsearchUser: {
          username,
          email
        }
      });
      try {
        const response = await http.post('/internal/app_search/single_user_role_mapping', {
          body
        });
        actions.setSingleUserRoleMapping(response);
        actions.setUserCreated();
        actions.initializeRoleMappings();
      } catch (e) {
        var _e$body2, _e$body2$attributes;
        actions.setRoleMappingErrors(e === null || e === void 0 ? void 0 : (_e$body2 = e.body) === null || _e$body2 === void 0 ? void 0 : (_e$body2$attributes = _e$body2.attributes) === null || _e$body2$attributes === void 0 ? void 0 : _e$body2$attributes.errors);
      }
    },
    closeUsersAndRolesFlyout: () => {
      (0, _flash_messages.clearFlashMessages)();
      const firstUser = values.elasticsearchUsers[0];
      actions.setElasticsearchUser(firstUser);
    },
    openRoleMappingFlyout: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    openSingleUserRoleMappingFlyout: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    setUserExistingRadioValue: ({
      userFormUserIsExisting
    }) => {
      const firstUser = values.elasticsearchUsers[0];
      actions.setElasticsearchUser(userFormUserIsExisting ? firstUser : emptyUser);
    },
    handleUsernameSelectChange: ({
      username
    }) => {
      const user = values.elasticsearchUsers.find(u => u.username === username);
      if (user) actions.setElasticsearchUser(user);
    }
  })
});