"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defaultApiToken = exports.CredentialsLogic = void 0;
var _kea = require("kea");
var _constants = require("../../../shared/constants");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _table_pagination = require("../../../shared/table_pagination");
var _app_logic = require("../../app_logic");
var _format_api_name = require("../../utils/format_api_name");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultApiToken = exports.defaultApiToken = {
  name: '',
  type: _constants2.ApiTokenTypes.Private,
  read: true,
  write: true,
  access_all_engines: true
};
// If we leave this inline, Prettier does some horrifying indenting nonsense :/

const CredentialsLogic = exports.CredentialsLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'credentials_logic'],
  actions: () => ({
    addEngineName: engineName => engineName,
    onApiTokenCreateSuccess: apiToken => apiToken,
    onApiTokenError: formErrors => formErrors,
    onApiTokenUpdateSuccess: apiToken => apiToken,
    removeEngineName: engineName => engineName,
    setAccessAllEngines: accessAll => accessAll,
    setCredentialsData: (meta, apiTokens) => ({
      meta,
      apiTokens
    }),
    setCredentialsDetails: details => details,
    setNameInputBlurred: nameInputBlurred => nameInputBlurred,
    setTokenReadWrite: ({
      name,
      checked
    }) => ({
      name,
      checked
    }),
    setTokenName: name => name,
    setTokenType: tokenType => tokenType,
    showCredentialsForm: (apiToken = {
      ...defaultApiToken
    }) => apiToken,
    hideCredentialsForm: false,
    resetCredentials: false,
    onPaginate: newPageIndex => ({
      newPageIndex
    }),
    fetchCredentials: true,
    fetchDetails: true,
    deleteApiKey: tokenName => tokenName,
    onApiTokenChange: () => null,
    onEngineSelect: engineName => engineName
  }),
  reducers: () => ({
    apiTokens: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setCredentialsData: (_, {
        apiTokens
      }) => apiTokens,
      // @ts-expect-error upgrade typescript v5.1.6
      onApiTokenCreateSuccess: (apiTokens, apiToken) => [...apiTokens, apiToken],
      // @ts-expect-error upgrade typescript v5.1.6
      onApiTokenUpdateSuccess: (apiTokens, apiToken) => [
      // @ts-expect-error upgrade typescript v5.1.6
      ...apiTokens.filter(token => token.name !== apiToken.name), apiToken]
    }],
    meta: [_constants.DEFAULT_META, {
      // @ts-expect-error upgrade typescript v5.1.6
      setCredentialsData: (_, {
        meta
      }) => meta,
      // @ts-expect-error upgrade typescript v5.1.6
      onPaginate: (state, {
        newPageIndex
      }) => (0, _table_pagination.updateMetaPageIndex)(state, newPageIndex)
    }],
    isCredentialsDetailsComplete: [false, {
      setCredentialsDetails: () => true,
      resetCredentials: () => false
    }],
    isCredentialsDataComplete: [false, {
      setCredentialsData: () => true,
      fetchCredentials: () => false,
      resetCredentials: () => false
    }],
    engines: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setCredentialsDetails: (_, {
        engines
      }) => engines
    }],
    nameInputBlurred: [false, {
      // @ts-expect-error upgrade typescript v5.1.6
      setNameInputBlurred: (_, nameInputBlurred) => nameInputBlurred
    }],
    activeApiToken: [defaultApiToken, {
      // @ts-expect-error upgrade typescript v5.1.6
      addEngineName: (activeApiToken, engineName) => ({
        ...activeApiToken,
        engines: [...(activeApiToken.engines || []), engineName]
      }),
      // @ts-expect-error upgrade typescript v5.1.6
      removeEngineName: (activeApiToken, engineName) => ({
        ...activeApiToken,
        // @ts-expect-error upgrade typescript v5.1.6
        engines: (activeApiToken.engines || []).filter(name => name !== engineName)
      }),
      // @ts-expect-error upgrade typescript v5.1.6
      setAccessAllEngines: (activeApiToken, accessAll) => ({
        ...activeApiToken,
        access_all_engines: accessAll,
        engines: accessAll ? [] : activeApiToken.engines
      }),
      onApiTokenCreateSuccess: () => defaultApiToken,
      onApiTokenUpdateSuccess: () => defaultApiToken,
      // @ts-expect-error upgrade typescript v5.1.6
      setTokenName: (activeApiToken, name) => ({
        ...activeApiToken,
        name: (0, _format_api_name.formatApiName)(name)
      }),
      // @ts-expect-error upgrade typescript v5.1.6
      setTokenReadWrite: (activeApiToken, {
        name,
        checked
      }) => ({
        ...activeApiToken,
        [name]: checked
      }),
      // @ts-expect-error upgrade typescript v5.1.6
      setTokenType: (activeApiToken, tokenType) => ({
        ...activeApiToken,
        access_all_engines: tokenType === _constants2.ApiTokenTypes.Admin ? false : activeApiToken.access_all_engines,
        engines: tokenType === _constants2.ApiTokenTypes.Admin ? [] : activeApiToken.engines,
        write: tokenType === _constants2.ApiTokenTypes.Private,
        read: tokenType === _constants2.ApiTokenTypes.Private,
        type: tokenType
      }),
      // @ts-expect-error upgrade typescript v5.1.6
      showCredentialsForm: (_, activeApiToken) => activeApiToken
    }],
    activeApiTokenRawName: ['', {
      // @ts-expect-error upgrade typescript v5.1.6
      setTokenName: (_, activeApiTokenRawName) => activeApiTokenRawName,
      // @ts-expect-error upgrade typescript v5.1.6
      showCredentialsForm: (_, activeApiToken) => activeApiToken.name,
      hideCredentialsForm: () => '',
      onApiTokenCreateSuccess: () => '',
      onApiTokenUpdateSuccess: () => ''
    }],
    shouldShowCredentialsForm: [false, {
      showCredentialsForm: () => true,
      hideCredentialsForm: () => false,
      onApiTokenCreateSuccess: () => false,
      onApiTokenUpdateSuccess: () => false
    }],
    formErrors: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      onApiTokenError: (_, formErrors) => formErrors,
      onApiTokenCreateSuccess: () => [],
      showCredentialsForm: () => [],
      resetCredentials: () => []
    }]
  }),
  selectors: ({
    selectors
  }) => ({
    fullEngineAccessChecked: [() => [_app_logic.AppLogic.selectors.myRole, selectors.activeApiToken], (myRole, activeApiToken) => !!(myRole.canAccessAllEngines && activeApiToken.access_all_engines)],
    dataLoading: [() => [selectors.isCredentialsDetailsComplete, selectors.isCredentialsDataComplete], (isCredentialsDetailsComplete, isCredentialsDataComplete) => {
      return isCredentialsDetailsComplete === false && isCredentialsDataComplete === false;
    }],
    activeApiTokenExists: [() => [selectors.activeApiToken], activeApiToken => !!activeApiToken.id]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    showCredentialsForm: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    fetchCredentials: async () => {
      try {
        const {
          http
        } = _http.HttpLogic.values;
        const {
          meta
        } = values;
        const query = {
          'page[current]': meta.page.current,
          'page[size]': meta.page.size
        };
        const response = await http.get('/internal/app_search/credentials', {
          query
        });
        actions.setCredentialsData(response.meta, response.results);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    fetchDetails: async () => {
      try {
        const {
          http
        } = _http.HttpLogic.values;
        const response = await http.get('/internal/app_search/credentials/details');
        actions.setCredentialsDetails(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    deleteApiKey: async tokenName => {
      try {
        const {
          http
        } = _http.HttpLogic.values;
        await http.delete(`/internal/app_search/credentials/${tokenName}`);
        actions.fetchCredentials();
        (0, _flash_messages.flashSuccessToast)((0, _constants2.DELETE_MESSAGE)(tokenName));
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    onApiTokenChange: async () => {
      const {
        id,
        name,
        engines,
        type,
        read,
        write
      } = values.activeApiToken;
      const data = {
        name,
        type
      };
      if (type === _constants2.ApiTokenTypes.Private) {
        data.read = read;
        data.write = write;
      }
      if (type !== _constants2.ApiTokenTypes.Admin) {
        data.access_all_engines = values.fullEngineAccessChecked;
        data.engines = engines;
      }
      try {
        const {
          http
        } = _http.HttpLogic.values;
        const body = JSON.stringify(data);
        if (id) {
          const response = await http.put(`/internal/app_search/credentials/${name}`, {
            body
          });
          actions.onApiTokenUpdateSuccess(response);
          (0, _flash_messages.flashSuccessToast)((0, _constants2.UPDATE_MESSAGE)(name));
        } else {
          const response = await http.post('/internal/app_search/credentials', {
            body
          });
          actions.onApiTokenCreateSuccess(response);
          (0, _flash_messages.flashSuccessToast)((0, _constants2.CREATE_MESSAGE)(name));
        }
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    onEngineSelect: engineName => {
      var _values$activeApiToke, _values$activeApiToke2;
      if ((_values$activeApiToke = values.activeApiToken) !== null && _values$activeApiToke !== void 0 && (_values$activeApiToke2 = _values$activeApiToke.engines) !== null && _values$activeApiToke2 !== void 0 && _values$activeApiToke2.includes(engineName)) {
        actions.removeEngineName(engineName);
      } else {
        actions.addEngineName(engineName);
      }
    }
  })
});