"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AddAnalyticsCollectionLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _api = require("../../../../../common/types/api");
var _encode_path_params = require("../../../shared/encode_path_params");
var _flash_messages = require("../../../shared/flash_messages");
var _kibana = require("../../../shared/kibana");
var _add_analytics_collection_api_logic = require("../../api/add_analytics_collection/add_analytics_collection_api_logic");
var _routes = require("../../routes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SERVER_ERROR_CODE = 500;
const NAME_VALIDATION = new RegExp(/^[a-z0-9\-]+$/);
const AddAnalyticsCollectionLogic = exports.AddAnalyticsCollectionLogic = (0, _kea.kea)({
  actions: {
    createAnalyticsCollection: () => {},
    setInputError: inputError => ({
      inputError
    }),
    setNameValue: name => ({
      name
    })
  },
  connect: {
    actions: [_add_analytics_collection_api_logic.AddAnalyticsCollectionsAPILogic, ['apiError', 'apiSuccess', 'makeRequest']],
    values: [_add_analytics_collection_api_logic.AddAnalyticsCollectionsAPILogic, ['status', 'error']]
  },
  listeners: ({
    values,
    actions
  }) => ({
    apiError: async error => {
      if (values.isSystemError) {
        var _error$body;
        if (error !== null && error !== void 0 && (_error$body = error.body) !== null && _error$body !== void 0 && _error$body.message) {
          _flash_messages.FlashMessagesLogic.actions.setFlashMessages([{
            description: error.body.message,
            message: _i18n.i18n.translate('xpack.enterpriseSearch.analytics.collectionsCreate.action.systemErrorMessage', {
              defaultMessage: 'Sorry, there was an error creating your collection.'
            }),
            type: 'error'
          }]);
        } else {
          (0, _flash_messages.flashAPIErrors)(error);
        }
      } else {
        var _error$body2;
        actions.setInputError((error === null || error === void 0 ? void 0 : (_error$body2 = error.body) === null || _error$body2 === void 0 ? void 0 : _error$body2.message) || null);
      }
    },
    apiSuccess: async ({
      name
    }) => {
      (0, _flash_messages.flashSuccessToast)(_i18n.i18n.translate('xpack.enterpriseSearch.analytics.collectionsCreate.action.successMessage', {
        defaultMessage: "Successfully added collection ''{name}''",
        values: {
          name
        }
      }));
      _kibana.KibanaLogic.values.navigateToUrl((0, _encode_path_params.generateEncodedPath)(_routes.COLLECTION_OVERVIEW_PATH, {
        name
      }));
    },
    createAnalyticsCollection: () => {
      const {
        name
      } = values;
      actions.makeRequest({
        name
      });
    },
    setNameValue: ({
      name
    }) => {
      if (!NAME_VALIDATION.test(name)) {
        actions.setInputError(_i18n.i18n.translate('xpack.enterpriseSearch.analytics.collectionsCreate.invalidName', {
          defaultMessage: 'Collection name can only contain lowercase letters, numbers, and hyphens'
        }));
      }
    }
  }),
  path: ['enterprise_search', 'analytics', 'add_analytics_collection'],
  reducers: {
    inputError: [null, {
      // @ts-expect-error upgrade typescript v5.1.6
      setInputError: (_, {
        inputError
      }) => inputError,
      setNameValue: () => null
    }],
    name: ['', {
      // @ts-expect-error upgrade typescript v5.1.6
      setNameValue: (_, {
        name
      }) => name
    }]
  },
  selectors: ({
    selectors
  }) => ({
    canSubmit: [() => [selectors.isLoading, selectors.name, selectors.inputError], (isLoading, name, inputError) => !isLoading && name.length > 0 && !inputError],
    isLoading: [() => [selectors.status], status => status === _api.Status.LOADING],
    isSuccess: [() => [selectors.status], status => status === _api.Status.SUCCESS],
    isSystemError: [() => [selectors.status, selectors.error], (status, error) => {
      var _error$body3;
      return Boolean(status === _api.Status.ERROR && ((error === null || error === void 0 ? void 0 : (_error$body3 = error.body) === null || _error$body3 === void 0 ? void 0 : _error$body3.statusCode) || 0) >= SERVER_ERROR_CODE);
    }]
  })
});