"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postDefendInsightsRoute = void 0;
var _moment = _interopRequireDefault(require("moment/moment"));
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securityAiPrompts = require("@kbn/security-ai-prompts");
var _tool_prompts = require("../../lib/prompt/tool_prompts");
var _build_response = require("../../lib/build_response");
var _helpers = require("../helpers");
var _helpers2 = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ROUTE_HANDLER_TIMEOUT = 10 * 60 * 1000; // 10 * 60 seconds = 10 minutes
const LANG_CHAIN_TIMEOUT = ROUTE_HANDLER_TIMEOUT - 10_000; // 9 minutes 50 seconds
const CONNECTOR_TIMEOUT = LANG_CHAIN_TIMEOUT - 10_000; // 9 minutes 40 seconds

const postDefendInsightsRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _elasticAssistantCommon.DEFEND_INSIGHTS,
    options: {
      timeout: {
        idleSocket: ROUTE_HANDLER_TIMEOUT
      }
    },
    security: {
      authz: {
        requiredPrivileges: ['securitySolution-writeWorkflowInsights']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.DefendInsightsPostRequestBody)
      },
      response: {
        200: {
          body: {
            custom: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.DefendInsightsPostResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const startTime = (0, _moment.default)(); // start timing the generation
    const resp = (0, _build_response.buildResponse)(response);
    const ctx = await context.resolve(['licensing', 'elasticAssistant']);
    const savedObjectsClient = ctx.elasticAssistant.savedObjectsClient;
    const assistantContext = ctx.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    try {
      const isEnabled = (0, _helpers2.isDefendInsightsEnabled)({
        request,
        logger,
        assistantContext
      });
      if (!isEnabled) {
        return response.notFound();
      }
      if (!ctx.licensing.license.hasAtLeast('enterprise')) {
        return response.forbidden({
          body: {
            message: 'Your license does not support Defend Workflows. Please upgrade your license.'
          }
        });
      }
      const actions = assistantContext.actions;
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const dataClient = await assistantContext.getDefendInsightsDataClient();
      const authenticatedUser = await assistantContext.getCurrentUser();
      if (authenticatedUser == null) {
        return resp.error({
          body: `Authenticated user not found`,
          statusCode: 401
        });
      }
      if (!dataClient) {
        return resp.error({
          body: `Defend insights data client not initialized`,
          statusCode: 500
        });
      }
      const pluginName = (0, _helpers.getPluginNameFromRequest)({
        request,
        defaultPluginName: _helpers.DEFAULT_PLUGIN_NAME,
        logger
      });
      const assistantTool = (0, _helpers2.getAssistantTool)(assistantContext.getRegisteredTools, pluginName);
      if (!assistantTool) {
        return response.notFound();
      }
      const {
        endpointIds,
        insightType,
        apiConfig,
        anonymizationFields,
        langSmithApiKey,
        langSmithProject,
        replacements
      } = request.body;
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      let latestReplacements = {
        ...replacements
      };
      const onNewReplacements = newReplacements => {
        latestReplacements = {
          ...latestReplacements,
          ...newReplacements
        };
      };
      const assistantToolParams = (0, _helpers2.getAssistantToolParams)({
        endpointIds,
        insightType,
        actionsClient,
        anonymizationFields,
        apiConfig,
        esClient,
        latestReplacements,
        contentReferencesStore: undefined,
        connectorTimeout: CONNECTOR_TIMEOUT,
        langChainTimeout: LANG_CHAIN_TIMEOUT,
        langSmithProject,
        langSmithApiKey,
        logger,
        onNewReplacements,
        request
      });
      const description = await (0, _securityAiPrompts.getPrompt)({
        actionsClient,
        connectorId: apiConfig.connectorId,
        localPrompts: _tool_prompts.localToolPrompts,
        promptId: assistantTool.name,
        promptGroupId: _tool_prompts.promptGroupId,
        savedObjectsClient
      });
      const toolInstance = assistantTool.getTool({
        ...assistantToolParams,
        description
      });
      const {
        currentInsight,
        defendInsightId
      } = await (0, _helpers2.createDefendInsight)(endpointIds, insightType, dataClient, authenticatedUser, apiConfig);
      toolInstance === null || toolInstance === void 0 ? void 0 : toolInstance.invoke('').then(rawDefendInsights => (0, _helpers2.updateDefendInsights)({
        apiConfig,
        defendInsightId,
        authenticatedUser,
        dataClient,
        latestReplacements,
        logger,
        rawDefendInsights,
        startTime,
        telemetry
      })).catch(err => (0, _helpers2.handleToolError)({
        apiConfig,
        defendInsightId,
        authenticatedUser,
        dataClient,
        err,
        latestReplacements,
        logger,
        telemetry
      }));
      return response.ok({
        body: currentInsight
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.postDefendInsightsRoute = postDefendInsightsRoute;