"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAttackDiscoveryRoute = void 0;
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _helpers = require("../helpers/helpers");
var _constants = require("../../../../common/constants");
var _build_response = require("../../../lib/build_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAttackDiscoveryRoute = router => {
  router.versioned.get({
    access: 'internal',
    path: _constants.ATTACK_DISCOVERY_BY_CONNECTOR_ID,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        params: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryGetRequestParams)
      },
      response: {
        200: {
          body: {
            custom: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryGetResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;
    try {
      const dataClient = await assistantContext.getAttackDiscoveryDataClient();
      const authenticatedUser = await assistantContext.getCurrentUser();
      const connectorId = decodeURIComponent(request.params.connectorId);
      if (authenticatedUser == null) {
        return resp.error({
          body: `Authenticated user not found`,
          statusCode: 401
        });
      }
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      const attackDiscovery = await (0, _helpers.updateAttackDiscoveryLastViewedAt)({
        dataClient,
        connectorId,
        authenticatedUser
      });
      const stats = await (0, _helpers.getAttackDiscoveryStats)({
        dataClient,
        authenticatedUser
      });
      return response.ok({
        body: attackDiscovery != null ? {
          data: attackDiscovery,
          stats
        } : {
          stats
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.getAttackDiscoveryRoute = getAttackDiscoveryRoute;