"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDisplayedColumns = getDisplayedColumns;
exports.getTimeColumn = getTimeColumn;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns properties necessary to display the time column
 * If it's an DataView with timefield, the time column is
 * prepended, not moveable and removeable
 * @param timeFieldName
 */
function getTimeColumn(timeFieldName) {
  return {
    name: timeFieldName,
    displayName: timeFieldName,
    isSortable: true,
    isRemoveable: false,
    colLeftIdx: -1,
    colRightIdx: -1
  };
}
/**
 * A given array of column names returns an array of properties
 * necessary to display the columns. If the given dataView
 * has a timefield, a time column is prepended
 * @param columns
 * @param dataView
 * @param hideTimeField
 * @param isShortDots
 */
function getDisplayedColumns(columns, dataView, hideTimeField, isShortDots) {
  if (!Array.isArray(columns) || typeof dataView !== 'object' || !dataView.getFieldByName) {
    return [];
  }
  const columnProps = columns.length === 0 ? [{
    name: '__document__',
    displayName: _i18n.i18n.translate('discover.docTable.tableHeader.documentHeader', {
      defaultMessage: 'Document'
    }),
    isSortable: false,
    isRemoveable: false,
    colLeftIdx: -1,
    colRightIdx: -1
  }] : columns.map((column, idx) => {
    var _field$displayName;
    const field = dataView.getFieldByName(column);
    return {
      name: column,
      displayName: (_field$displayName = field === null || field === void 0 ? void 0 : field.displayName) !== null && _field$displayName !== void 0 ? _field$displayName : column,
      isSortable: !!(field && field.sortable),
      isRemoveable: column !== '_source' || columns.length > 1,
      colLeftIdx: idx - 1 < 0 ? -1 : idx - 1,
      colRightIdx: idx + 1 >= columns.length ? -1 : idx + 1
    };
  });
  return !hideTimeField && dataView.timeFieldName ? [getTimeColumn(dataView.timeFieldName), ...columnProps] : columnProps;
}