"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getBooleanFieldsStatsRequest = exports.fetchBooleanFieldsStats = void 0;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _utils = require("./utils");
var _build_random_sampler_agg = require("./build_random_sampler_agg");
var _field_stats = require("../../../../../common/types/field_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getBooleanFieldsStatsRequest = (params, fields) => {
  const {
    index,
    query,
    runtimeFieldMap
  } = params;
  const size = 0;
  const aggs = {};
  fields.forEach((field, i) => {
    const safeFieldName = field.safeFieldName;
    aggs[`${safeFieldName}_value_count`] = {
      filter: {
        exists: {
          field: field.fieldName
        }
      }
    };
    aggs[`${safeFieldName}_values`] = {
      terms: {
        field: field.fieldName,
        size: 2
      }
    };
  });
  const searchBody = {
    query,
    aggs: (0, _build_random_sampler_agg.buildAggregationWithSamplingOption)(aggs, params.samplingOption),
    ...((0, _mlIsPopulatedObject.isPopulatedObject)(runtimeFieldMap) ? {
      runtime_mappings: runtimeFieldMap
    } : {})
  };
  return {
    index,
    size,
    body: searchBody
  };
};
exports.getBooleanFieldsStatsRequest = getBooleanFieldsStatsRequest;
const fetchBooleanFieldsStats = (dataSearch, params, fields, options) => {
  const request = getBooleanFieldsStatsRequest(params, fields);
  return dataSearch.search({
    params: request
  }, options).pipe((0, _rxjs.catchError)(e => (0, _rxjs.of)({
    fields,
    error: (0, _mlErrorUtils.extractErrorProperties)(e)
  })), (0, _rxjs.map)(resp => {
    if (!(0, _field_stats.isIKibanaSearchResponse)(resp)) return resp;
    const aggregations = resp.rawResponse.aggregations;
    const aggsPath = ['sample'];
    const sampleCount = (0, _lodash.get)(aggregations, [...aggsPath, 'doc_count'], 0);
    const batchStats = fields.map((field, i) => {
      const safeFieldName = field.fieldName;
      // Sampler agg will yield doc_count that's bigger than the actual # of sampled records
      // because it uses the stored _doc_count if available
      // https://www.elastic.co/guide/en/elasticsearch/reference/current/mapping-doc-count-field.html
      // therefore we need to correct it by multiplying by the sampled probability
      const count = (0, _lodash.get)(aggregations, [...aggsPath, `${safeFieldName}_value_count`, 'doc_count'], 0);
      const fieldAgg = (0, _lodash.get)(aggregations, [...aggsPath, `${safeFieldName}_values`], {});
      const {
        topValuesSampleSize,
        topValues
      } = (0, _utils.processTopValues)(fieldAgg);
      const multiplier = count > sampleCount ? (0, _lodash.get)(aggregations, [...aggsPath, 'probability'], 1) : 1;
      const stats = {
        fieldName: field.fieldName,
        count: count * multiplier,
        trueCount: 0,
        falseCount: 0,
        topValues,
        topValuesSampleSize
      };
      const valueBuckets = (0, _lodash.get)(aggregations, [...aggsPath, `${safeFieldName}_values`, 'buckets'], []);
      valueBuckets.forEach(bucket => {
        stats[`${bucket.key_as_string}Count`] = bucket.doc_count;
      });
      return stats;
    });
    return batchStats;
  }));
};
exports.fetchBooleanFieldsStats = fetchBooleanFieldsStats;