"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAutoRefreshLoop = void 0;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _page_visibility = require("./page_visibility");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Creates a loop for timepicker's auto refresh
 * It has a "confirmation" mechanism:
 * When auto refresh loop emits, it won't continue automatically,
 * until each subscriber calls received `done` function.
 *
 * Also, it will pause when the page is not visible.
 *
 * @internal
 */
const createAutoRefreshLoop = () => {
  let subscribersCount = 0;
  const tick = new _rxjs.Subject();
  let _timeoutHandle;
  let _timeout = 0;
  function start() {
    stop();
    if (_timeout === 0) return;
    const timeoutHandle = window.setTimeout(() => {
      let pendingDoneCount = subscribersCount;
      const done = () => {
        if (timeoutHandle !== _timeoutHandle) return;
        pendingDoneCount--;
        if (pendingDoneCount === 0) {
          start();
        }
      };
      tick.next(done);
    }, _timeout);
    _timeoutHandle = timeoutHandle;
  }
  function stop() {
    window.clearTimeout(_timeoutHandle);
    _timeoutHandle = -1;
  }
  const pageVisible$ = (0, _page_visibility.createPageVisibility$)().pipe((0, _rxjs.filter)(visibility => visibility === 'visible'));
  const tickWhenVisible$ = tick.pipe((0, _rxjs.delayWhen)(() => pageVisible$));
  return {
    stop: () => {
      _timeout = 0;
      stop();
    },
    start: timeout => {
      _timeout = timeout;
      if (subscribersCount > 0) {
        start();
      }
    },
    loop$: (0, _rxjs.defer)(() => {
      subscribersCount++;
      start(); // restart the loop on a new subscriber
      return tickWhenVisible$.pipe((0, _rxjs.map)(doneCb => (0, _lodash.once)(doneCb))); // each subscriber allowed to call done only once
    }).pipe((0, _rxjs.finalize)(() => {
      subscribersCount--;
      if (subscribersCount === 0) {
        stop();
      } else {
        start(); // restart the loop to potentially unblock the interval
      }
    }))
  };
};
exports.createAutoRefreshLoop = createAutoRefreshLoop;