"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEsqlFn = void 0;
var _fieldTypes = require("@kbn/field-types");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/inspector-plugin/common");
var _esqlUtils = require("@kbn/esql-utils");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _esQuery = require("@kbn/es-query");
var _es_query = require("../../es_query");
var _query = require("../../query");
var _2 = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function extractTypeAndReason(attributes) {
  if (['type', 'reason'].every(prop => prop in attributes)) {
    return attributes;
  }
  if ('error' in attributes) {
    return extractTypeAndReason(attributes.error);
  }
  return {};
}
const getEsqlFn = ({
  getStartDependencies
}) => {
  const essql = {
    name: 'esql',
    type: 'datatable',
    inputTypes: ['kibana_context', 'null'],
    help: _i18n.i18n.translate('data.search.esql.help', {
      defaultMessage: 'Queries Elasticsearch using ES|QL.'
    }),
    args: {
      query: {
        aliases: ['_', 'q'],
        types: ['string'],
        help: _i18n.i18n.translate('data.search.esql.query.help', {
          defaultMessage: 'An ES|QL query.'
        })
      },
      // timezone: {
      //   aliases: ['tz'],
      //   types: ['string'],
      //   default: 'UTC',
      //   help: i18n.translate('data.search.esql.timezone.help', {
      //     defaultMessage:
      //       'The timezone to use for date operations. Valid ISO8601 formats and UTC offsets both work.',
      //   }),
      // },
      timeField: {
        aliases: ['timeField'],
        types: ['string'],
        help: _i18n.i18n.translate('data.search.essql.timeField.help', {
          defaultMessage: 'The time field to use in the time range filter set in the context.'
        })
      },
      locale: {
        aliases: ['locale'],
        types: ['string'],
        help: _i18n.i18n.translate('data.search.essql.locale.help', {
          defaultMessage: 'The locale to use.'
        })
      },
      titleForInspector: {
        aliases: ['titleForInspector'],
        types: ['string'],
        help: _i18n.i18n.translate('data.search.esql.titleForInspector.help', {
          defaultMessage: 'The title to show in Inspector.'
        })
      },
      descriptionForInspector: {
        aliases: ['descriptionForInspector'],
        types: ['string'],
        help: _i18n.i18n.translate('data.search.esql.descriptionForInspector.help', {
          defaultMessage: 'The description to show in Inspector.'
        })
      }
    },
    allowCache: {
      withSideEffects: (_, {
        inspectorAdapters
      }) => {
        return (0, _2.getSideEffectFunction)(inspectorAdapters);
      }
    },
    fn(input, {
      query,
      /* timezone, */timeField,
      locale,
      titleForInspector,
      descriptionForInspector
    }, {
      abortSignal,
      inspectorAdapters,
      getKibanaRequest
    }) {
      return (0, _rxjs.defer)(() => getStartDependencies(() => {
        const request = getKibanaRequest === null || getKibanaRequest === void 0 ? void 0 : getKibanaRequest();
        if (!request) {
          throw new Error('A KibanaRequest is required to run queries on the server. ' + 'Please provide a request object to the expression execution params.');
        }
        return request;
      })).pipe((0, _rxjs.switchMap)(({
        search,
        uiSettings
      }) => {
        const params = {
          query,
          // time_zone: timezone,
          locale,
          include_ccs_metadata: true
        };
        if (input) {
          var _input$filters;
          const esQueryConfigs = (0, _es_query.getEsQueryConfig)(uiSettings);
          const namedParams = (0, _esqlUtils.getNamedParams)(query, input.timeRange, input.esqlVariables);
          if (namedParams.length) {
            params.params = namedParams;
          }
          const timeFilter = input.timeRange && (0, _query.getTime)(undefined, input.timeRange, {
            fieldName: timeField
          });

          // Used for debugging & inside automated tests to simulate a slow query
          const delayFilter = window.ELASTIC_ESQL_DELAY_SECONDS ? {
            meta: {},
            query: {
              error_query: {
                indices: [{
                  name: '*',
                  error_type: 'warning',
                  stall_time_seconds: window.ELASTIC_ESQL_DELAY_SECONDS
                }]
              }
            }
          } : undefined;
          const filters = [...((_input$filters = input.filters) !== null && _input$filters !== void 0 ? _input$filters : []), ...(timeFilter ? [timeFilter] : []), ...(delayFilter ? [delayFilter] : [])];
          params.filter = (0, _esQuery.buildEsQuery)(undefined, input.query || [], filters, esQueryConfigs);
        }
        let startTime = Date.now();
        const logInspectorRequest = () => {
          if (!inspectorAdapters.requests) {
            inspectorAdapters.requests = new _common.RequestAdapter();
          }
          const request = inspectorAdapters.requests.start(titleForInspector !== null && titleForInspector !== void 0 ? titleForInspector : _i18n.i18n.translate('data.search.dataRequest.title', {
            defaultMessage: 'Data'
          }), {
            description: descriptionForInspector !== null && descriptionForInspector !== void 0 ? descriptionForInspector : _i18n.i18n.translate('data.search.es_search.dataRequest.description', {
              defaultMessage: 'This request queries Elasticsearch to fetch the data for the visualization.'
            })
          }, startTime);
          startTime = Date.now();
          return request;
        };
        return search({
          params: {
            ...params,
            dropNullColumns: true
          }
        }, {
          abortSignal,
          strategy: _2.ESQL_ASYNC_SEARCH_STRATEGY
        }).pipe((0, _rxjs.catchError)(error => {
          if (!error.attributes) {
            error.message = `Unexpected error from Elasticsearch: ${error.message}`;
          } else {
            const {
              type,
              reason
            } = extractTypeAndReason(error.attributes);
            if (type === 'parsing_exception') {
              error.message = `Couldn't parse Elasticsearch ES|QL query. Check your query and try again. Error: ${reason}`;
            } else {
              error.message = `Unexpected error from Elasticsearch: ${type} - ${reason}`;
            }
          }
          return (0, _rxjs.throwError)(() => error);
        }), (0, _rxjs.tap)({
          next(response) {
            if ((0, _2.isRunningResponse)(response)) return;
            const {
              rawResponse,
              requestParams
            } = response;
            logInspectorRequest().stats({
              hits: {
                label: _i18n.i18n.translate('data.search.es_search.hitsLabel', {
                  defaultMessage: 'Hits'
                }),
                value: `${rawResponse.values.length}`,
                description: _i18n.i18n.translate('data.search.es_search.hitsDescription', {
                  defaultMessage: 'The number of documents returned by the query.'
                })
              },
              ...((rawResponse === null || rawResponse === void 0 ? void 0 : rawResponse.took) && {
                queryTime: {
                  label: _i18n.i18n.translate('data.search.es_search.queryTimeLabel', {
                    defaultMessage: 'Query time'
                  }),
                  value: _i18n.i18n.translate('data.search.es_search.queryTimeValue', {
                    defaultMessage: '{queryTime}ms',
                    values: {
                      queryTime: rawResponse.took
                    }
                  }),
                  description: _i18n.i18n.translate('data.search.es_search.queryTimeDescription', {
                    defaultMessage: 'The time it took to process the query. ' + 'Does not include the time to send the request or parse it in the browser.'
                  })
                }
              })
            }).json(params).ok({
              json: {
                rawResponse
              },
              requestParams
            });
          },
          error(error) {
            logInspectorRequest().json(params).error({
              json: 'attributes' in error ? error.attributes : {
                message: error.message
              }
            });
          }
        }));
      }), (0, _rxjs.map)(({
        rawResponse: body,
        warning
      }) => {
        var _body$all_columns, _body$columns, _map, _ref, _body$all_columns2, _input$esqlVariables;
        // all_columns in the response means that there is a separation between
        // columns with data and empty columns
        // columns contain only columns with data while all_columns everything
        const hasEmptyColumns = body.all_columns && ((_body$all_columns = body.all_columns) === null || _body$all_columns === void 0 ? void 0 : _body$all_columns.length) > body.columns.length;
        const lookup = new Set(hasEmptyColumns ? ((_body$columns = body.columns) === null || _body$columns === void 0 ? void 0 : _body$columns.map(({
          name
        }) => name)) || [] : []);
        const allColumns = (_map = (_ref = (_body$all_columns2 = body.all_columns) !== null && _body$all_columns2 !== void 0 ? _body$all_columns2 : body.columns) === null || _ref === void 0 ? void 0 : _ref.map(({
          name,
          type
        }) => ({
          id: name,
          name,
          meta: {
            type: (0, _fieldTypes.esFieldTypeToKibanaFieldType)(type),
            esType: type,
            sourceParams: type === 'date' ? {
              appliedTimeRange: input === null || input === void 0 ? void 0 : input.timeRange,
              params: {}
            } : {}
          },
          isNull: hasEmptyColumns ? !lookup.has(name) : false
        }))) !== null && _map !== void 0 ? _map : [];
        const updatedWithVariablesColumns = (0, _esqlUtils.mapVariableToColumn)(query, (_input$esqlVariables = input === null || input === void 0 ? void 0 : input.esqlVariables) !== null && _input$esqlVariables !== void 0 ? _input$esqlVariables : [], allColumns);

        // sort only in case of empty columns to correctly align columns to items in values array
        if (hasEmptyColumns) {
          updatedWithVariablesColumns.sort((a, b) => Number(a.isNull) - Number(b.isNull));
        }
        const columnNames = updatedWithVariablesColumns === null || updatedWithVariablesColumns === void 0 ? void 0 : updatedWithVariablesColumns.map(({
          name
        }) => name);
        const rows = body.values.map(row => (0, _lodash.zipObject)(columnNames, row));
        return {
          type: 'datatable',
          meta: {
            type: _2.ESQL_TABLE_TYPE
          },
          columns: updatedWithVariablesColumns,
          rows,
          warning
        };
      }));
    }
  };
  return essql;
};
exports.getEsqlFn = getEsqlFn;