"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.saveDashboardState = void 0;
var _i18n = require("@kbn/i18n");
var _ = require("..");
var _telemetry_constants = require("../../../utils/telemetry_constants");
var _dashboard_backup_service = require("../../dashboard_backup_service");
var _kibana_services = require("../../kibana_services");
var _get_serialized_state = require("../../../dashboard_api/get_serialized_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const saveDashboardState = async ({
  controlGroupReferences,
  lastSavedId,
  saveOptions,
  dashboardState,
  panelReferences,
  searchSourceReferences
}) => {
  const dashboardContentManagementCache = (0, _.getDashboardContentManagementCache)();
  const {
    attributes,
    references
  } = (0, _get_serialized_state.getSerializedState)({
    controlGroupReferences,
    generateNewIds: saveOptions.saveAsCopy,
    dashboardState,
    panelReferences,
    searchSourceReferences
  });

  /**
   * Save the saved object using the content management
   */
  const idToSaveTo = saveOptions.saveAsCopy ? undefined : lastSavedId;
  try {
    const result = idToSaveTo ? await _kibana_services.contentManagementService.client.update({
      id: idToSaveTo,
      contentTypeId: _telemetry_constants.DASHBOARD_CONTENT_ID,
      data: attributes,
      options: {
        references,
        /** perform a "full" update instead, where the provided attributes will fully replace the existing ones */
        mergeAttributes: false
      }
    }) : await _kibana_services.contentManagementService.client.create({
      contentTypeId: _telemetry_constants.DASHBOARD_CONTENT_ID,
      data: attributes,
      options: {
        references
      }
    });
    const newId = result.item.id;
    if (newId) {
      _kibana_services.coreServices.notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('dashboard.dashboardWasSavedSuccessMessage', {
          defaultMessage: `Dashboard ''{title}'' was saved`,
          values: {
            title: dashboardState.title
          }
        }),
        className: 'eui-textBreakWord',
        'data-test-subj': 'saveDashboardSuccess'
      });

      /**
       * If the dashboard id has been changed, redirect to the new ID to keep the url param in sync.
       */
      if (newId !== lastSavedId) {
        (0, _dashboard_backup_service.getDashboardBackupService)().clearState(lastSavedId);
        return {
          redirectRequired: true,
          id: newId,
          references
        };
      } else {
        dashboardContentManagementCache.deleteDashboard(newId); // something changed in an existing dashboard, so delete it from the cache so that it can be re-fetched
      }
    }
    return {
      id: newId,
      references
    };
  } catch (error) {
    _kibana_services.coreServices.notifications.toasts.addDanger({
      title: _i18n.i18n.translate('dashboard.dashboardWasNotSavedDangerMessage', {
        defaultMessage: `Dashboard ''{title}'' was not saved. Error: {errorMessage}`,
        values: {
          title: dashboardState.title,
          errorMessage: error.message
        }
      }),
      'data-test-subj': 'saveDashboardFailure'
    });
    return {
      error
    };
  }
};
exports.saveDashboardState = saveDashboardState;