"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadDashboardState = void 0;
exports.migrateLegacyQuery = migrateLegacyQuery;
var _lodash = require("lodash");
var _uuid = require("uuid");
var _public = require("@kbn/data-plugin/public");
var _public2 = require("@kbn/kibana-utils-plugin/public");
var _clean_filters_for_serialize = require("../../../utils/clean_filters_for_serialize");
var _ = require("..");
var _common = require("../../../../common");
var _telemetry_constants = require("../../../utils/telemetry_constants");
var _default_dashboard_input = require("../../../dashboard_api/default_dashboard_input");
var _kibana_services = require("../../kibana_services");
var _dashboard_versioning = require("./dashboard_versioning");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function migrateLegacyQuery(query) {
  // Lucene was the only option before, so language-less queries are all lucene
  if (!(0, _lodash.has)(query, 'language')) {
    return {
      query,
      language: 'lucene'
    };
  }
  return query;
}
const loadDashboardState = async ({
  id
}) => {
  var _ref, _savedObjectsTaggingS, _savedObjectsTaggingS2;
  const {
    search: dataSearchService,
    query: {
      queryString
    }
  } = _kibana_services.dataService;
  const dashboardContentManagementCache = (0, _.getDashboardContentManagementCache)();
  const savedObjectId = id;
  const embeddableId = (0, _uuid.v4)();
  const newDashboardState = {
    ..._default_dashboard_input.DEFAULT_DASHBOARD_INPUT,
    id: embeddableId
  };

  /**
   * This is a newly created dashboard, so there is no saved object state to load.
   */
  if (!savedObjectId) {
    return {
      dashboardInput: newDashboardState,
      dashboardFound: true,
      newDashboardCreated: true,
      references: []
    };
  }

  /**
   * Load the saved object from Content Management
   */
  let rawDashboardContent;
  let resolveMeta;
  const cachedDashboard = dashboardContentManagementCache.fetchDashboard(id);
  if (cachedDashboard) {
    /** If the dashboard exists in the cache, use the cached version to load the dashboard */
    ({
      item: rawDashboardContent,
      meta: resolveMeta
    } = cachedDashboard);
  } else {
    /** Otherwise, fetch and load the dashboard from the content management client, and add it to the cache */
    const result = await _kibana_services.contentManagementService.client.get({
      contentTypeId: _telemetry_constants.DASHBOARD_CONTENT_ID,
      id
    }).catch(e => {
      throw new _public2.SavedObjectNotFound(_telemetry_constants.DASHBOARD_CONTENT_ID, id);
    });
    ({
      item: rawDashboardContent,
      meta: resolveMeta
    } = result);
    const {
      outcome: loadOutcome
    } = resolveMeta;
    if (loadOutcome !== 'aliasMatch') {
      /**
       * Only add the dashboard to the cache if it does not require a redirect - otherwise, the meta
       * alias info gets cached and prevents the dashboard contents from being updated
       */
      dashboardContentManagementCache.addDashboard(result);
    }
  }
  if (!rawDashboardContent || !rawDashboardContent.version) {
    return {
      dashboardInput: newDashboardState,
      dashboardFound: false,
      dashboardId: savedObjectId,
      references: []
    };
  }

  /**
   * Inject saved object references back into the saved object attributes
   */
  const {
    references,
    attributes: rawAttributes,
    managed
  } = rawDashboardContent;
  const attributes = (() => {
    if (!references || references.length === 0) return rawAttributes;
    return (0, _common.injectReferences)({
      references,
      attributes: rawAttributes
    }, {
      embeddablePersistableStateService: _kibana_services.embeddableService
    });
  })();

  /**
   * Create search source and pull filters and query from it.
   */
  let searchSourceValues = attributes.kibanaSavedObjectMeta.searchSource;
  const searchSource = await (async () => {
    if (!searchSourceValues) {
      return await dataSearchService.searchSource.create();
    }
    try {
      searchSourceValues = (0, _public.injectSearchSourceReferences)(searchSourceValues, references);
      return await dataSearchService.searchSource.create(searchSourceValues);
    } catch (error) {
      return await dataSearchService.searchSource.create();
    }
  })();
  const filters = (0, _clean_filters_for_serialize.cleanFiltersForSerialize)((_ref = searchSource === null || searchSource === void 0 ? void 0 : searchSource.getOwnField('filter')) !== null && _ref !== void 0 ? _ref : []);
  const query = migrateLegacyQuery((searchSource === null || searchSource === void 0 ? void 0 : searchSource.getOwnField('query')) || queryString.getDefaultQuery() // TODO SAVED DASHBOARDS determine if migrateLegacyQuery is still needed
  );
  const {
    refreshInterval,
    description,
    timeRestore,
    options,
    panels,
    timeFrom,
    version,
    timeTo,
    title
  } = attributes;
  const timeRange = timeRestore && timeFrom && timeTo ? {
    from: timeFrom,
    to: timeTo
  } : undefined;
  const panelMap = (0, _common.convertPanelsArrayToPanelMap)(panels !== null && panels !== void 0 ? panels : []);
  return {
    managed,
    references,
    resolveMeta,
    dashboardInput: {
      ..._default_dashboard_input.DEFAULT_DASHBOARD_INPUT,
      ...options,
      id: embeddableId,
      refreshInterval,
      timeRestore,
      description,
      timeRange,
      filters,
      panels: panelMap,
      query,
      title,
      viewMode: 'view',
      // dashboards loaded from saved object default to view mode. If it was edited recently, the view mode from session storage will override this.
      tags: (_savedObjectsTaggingS = _kibana_services.savedObjectsTaggingService === null || _kibana_services.savedObjectsTaggingService === void 0 ? void 0 : (_savedObjectsTaggingS2 = _kibana_services.savedObjectsTaggingService.getTaggingApi()) === null || _savedObjectsTaggingS2 === void 0 ? void 0 : _savedObjectsTaggingS2.ui.getTagIdsFromReferences(references)) !== null && _savedObjectsTaggingS !== void 0 ? _savedObjectsTaggingS : [],
      controlGroupInput: attributes.controlGroupInput,
      ...(version && {
        version: (0, _dashboard_versioning.convertNumberToDashboardVersion)(version)
      })
    },
    dashboardFound: true,
    dashboardId: savedObjectId
  };
};
exports.loadDashboardState = loadDashboardState;