"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializePanelsManager = initializePanelsManager;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _uuid = require("uuid");
var _std = require("@kbn/std");
var _analytics = require("@kbn/analytics");
var _public = require("@kbn/embeddable-plugin/public");
var _presentationPublishing = require("@kbn/presentation-publishing");
var _i18n = require("@kbn/i18n");
var _kibana_services = require("../services/kibana_services");
var _common = require("../../common");
var _dashboard_app_strings = require("../dashboard_app/_dashboard_app_strings");
var _place_new_panel_strategies = require("../dashboard_container/panel_placement/place_new_panel_strategies");
var _content_management = require("../../common/content_management");
var _telemetry_constants = require("../utils/telemetry_constants");
var _panel_placement_registry = require("../dashboard_container/panel_placement/panel_placement_registry");
var _are_panel_layouts_equal = require("./are_panel_layouts_equal");
var _dashboard_actions_strings = require("../dashboard_actions/_dashboard_actions_strings");
var _place_clone_panel_strategy = require("../dashboard_container/panel_placement/place_clone_panel_strategy");
var _plugin_constants = require("../plugin_constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function initializePanelsManager(incomingEmbeddable, initialPanels, initialPanelsRuntimeState, trackPanel, getReferencesForPanelId, pushReferences) {
  const children$ = new _rxjs.BehaviorSubject({});
  const panels$ = new _rxjs.BehaviorSubject(initialPanels);
  function setPanels(panels) {
    if (panels !== panels$.value) panels$.next(panels);
  }
  let restoredRuntimeState = initialPanelsRuntimeState;
  function setRuntimeStateForChild(childId, state) {
    restoredRuntimeState[childId] = state;
  }

  // --------------------------------------------------------------------------------------
  // Place the incoming embeddable if there is one
  // --------------------------------------------------------------------------------------
  if (incomingEmbeddable) {
    let incomingEmbeddablePanelState;
    if (incomingEmbeddable.embeddableId && Boolean(panels$.value[incomingEmbeddable.embeddableId])) {
      // this embeddable already exists, just update the explicit input.
      incomingEmbeddablePanelState = panels$.value[incomingEmbeddable.embeddableId];
      const sameType = incomingEmbeddablePanelState.type === incomingEmbeddable.type;
      incomingEmbeddablePanelState.type = incomingEmbeddable.type;
      setRuntimeStateForChild(incomingEmbeddable.embeddableId, {
        // if the incoming panel is the same type as what was there before we can safely spread the old panel's explicit input
        ...(sameType ? incomingEmbeddablePanelState.explicitInput : {}),
        ...incomingEmbeddable.input,
        id: incomingEmbeddable.embeddableId,
        // maintain hide panel titles setting.
        hidePanelTitles: incomingEmbeddablePanelState.explicitInput.hidePanelTitles
      });
      incomingEmbeddablePanelState.explicitInput = {
        id: incomingEmbeddablePanelState.explicitInput.id
      };
    } else {
      var _incomingEmbeddable$e, _incomingEmbeddable$s, _incomingEmbeddable$s2, _incomingEmbeddable$s3, _incomingEmbeddable$s4;
      // otherwise this incoming embeddable is brand new.
      const embeddableId = (_incomingEmbeddable$e = incomingEmbeddable.embeddableId) !== null && _incomingEmbeddable$e !== void 0 ? _incomingEmbeddable$e : (0, _uuid.v4)();
      setRuntimeStateForChild(embeddableId, incomingEmbeddable.input);
      const {
        newPanelPlacement
      } = (0, _place_new_panel_strategies.runPanelPlacementStrategy)(_plugin_constants.PanelPlacementStrategy.findTopLeftMostOpenSpace, {
        width: (_incomingEmbeddable$s = (_incomingEmbeddable$s2 = incomingEmbeddable.size) === null || _incomingEmbeddable$s2 === void 0 ? void 0 : _incomingEmbeddable$s2.width) !== null && _incomingEmbeddable$s !== void 0 ? _incomingEmbeddable$s : _content_management.DEFAULT_PANEL_WIDTH,
        height: (_incomingEmbeddable$s3 = (_incomingEmbeddable$s4 = incomingEmbeddable.size) === null || _incomingEmbeddable$s4 === void 0 ? void 0 : _incomingEmbeddable$s4.height) !== null && _incomingEmbeddable$s3 !== void 0 ? _incomingEmbeddable$s3 : _content_management.DEFAULT_PANEL_HEIGHT,
        currentPanels: panels$.value
      });
      incomingEmbeddablePanelState = {
        explicitInput: {
          id: embeddableId
        },
        type: incomingEmbeddable.type,
        gridData: {
          ...newPanelPlacement,
          i: embeddableId
        }
      };
    }
    setPanels({
      ...panels$.value,
      [incomingEmbeddablePanelState.explicitInput.id]: incomingEmbeddablePanelState
    });
    trackPanel.setScrollToPanelId(incomingEmbeddablePanelState.explicitInput.id);
    trackPanel.setHighlightPanelId(incomingEmbeddablePanelState.explicitInput.id);
  }
  async function untilEmbeddableLoaded(id) {
    if (!panels$.value[id]) {
      throw new _public.PanelNotFoundError();
    }
    if (children$.value[id]) {
      return children$.value[id];
    }
    return new Promise((resolve, reject) => {
      const subscription = (0, _rxjs.merge)(children$, panels$).subscribe(() => {
        if (children$.value[id]) {
          subscription.unsubscribe();
          resolve(children$.value[id]);
        }

        // If we hit this, the panel was removed before the embeddable finished loading.
        if (panels$.value[id] === undefined) {
          subscription.unsubscribe();
          resolve(undefined);
        }
      });
    });
  }
  function getDashboardPanelFromId(panelId) {
    const panel = panels$.value[panelId];
    const child = children$.value[panelId];
    if (!child || !panel) throw new _public.PanelNotFoundError();
    const serialized = (0, _presentationPublishing.apiHasSerializableState)(child) ? child.serializeState() : {
      rawState: {}
    };
    return {
      type: panel.type,
      explicitInput: {
        ...panel.explicitInput,
        ...serialized.rawState
      },
      gridData: panel.gridData,
      references: serialized.references
    };
  }
  async function getPanelTitles() {
    const titles = [];
    await (0, _std.asyncForEach)(Object.keys(panels$.value), async id => {
      const childApi = await untilEmbeddableLoaded(id);
      const title = (0, _presentationPublishing.apiPublishesTitle)(childApi) ? (0, _presentationPublishing.getTitle)(childApi) : '';
      if (title) titles.push(title);
    });
    return titles;
  }
  return {
    api: {
      addNewPanel: async (panelPackage, displaySuccessMessage) => {
        var _customPlacementSetti, _customPlacementSetti2, _customPlacementSetti3;
        const {
          panelType: type,
          serializedState,
          initialState
        } = panelPackage;
        _kibana_services.usageCollectionService === null || _kibana_services.usageCollectionService === void 0 ? void 0 : _kibana_services.usageCollectionService.reportUiCounter(_telemetry_constants.DASHBOARD_UI_METRIC_ID, _analytics.METRIC_TYPE.CLICK, type);
        const newId = (0, _uuid.v4)();
        const getCustomPlacementSettingFunc = (0, _panel_placement_registry.getDashboardPanelPlacementSetting)(type);
        const customPlacementSettings = getCustomPlacementSettingFunc ? await getCustomPlacementSettingFunc(initialState) : undefined;
        const {
          newPanelPlacement,
          otherPanels
        } = (0, _place_new_panel_strategies.runPanelPlacementStrategy)((_customPlacementSetti = customPlacementSettings === null || customPlacementSettings === void 0 ? void 0 : customPlacementSettings.strategy) !== null && _customPlacementSetti !== void 0 ? _customPlacementSetti : _plugin_constants.PanelPlacementStrategy.findTopLeftMostOpenSpace, {
          currentPanels: panels$.value,
          height: (_customPlacementSetti2 = customPlacementSettings === null || customPlacementSettings === void 0 ? void 0 : customPlacementSettings.height) !== null && _customPlacementSetti2 !== void 0 ? _customPlacementSetti2 : _content_management.DEFAULT_PANEL_HEIGHT,
          width: (_customPlacementSetti3 = customPlacementSettings === null || customPlacementSettings === void 0 ? void 0 : customPlacementSettings.width) !== null && _customPlacementSetti3 !== void 0 ? _customPlacementSetti3 : _content_management.DEFAULT_PANEL_WIDTH
        });
        if (serializedState !== null && serializedState !== void 0 && serializedState.references && serializedState.references.length > 0) {
          pushReferences((0, _common.prefixReferencesFromPanel)(newId, serializedState.references));
        }
        const newPanel = {
          type,
          gridData: {
            ...newPanelPlacement,
            i: newId
          },
          explicitInput: {
            ...(serializedState === null || serializedState === void 0 ? void 0 : serializedState.rawState),
            id: newId
          }
        };
        if (initialState) setRuntimeStateForChild(newId, initialState);
        setPanels({
          ...otherPanels,
          [newId]: newPanel
        });
        if (displaySuccessMessage) {
          _kibana_services.coreServices.notifications.toasts.addSuccess({
            title: (0, _dashboard_app_strings.getPanelAddedSuccessString)(newPanel.explicitInput.title),
            'data-test-subj': 'addEmbeddableToDashboardSuccess'
          });
          trackPanel.setScrollToPanelId(newId);
          trackPanel.setHighlightPanelId(newId);
        }
        return await untilEmbeddableLoaded(newId);
      },
      canRemovePanels: () => trackPanel.expandedPanelId$.value === undefined,
      children$,
      duplicatePanel: async idToDuplicate => {
        var _getTitle;
        const panelToClone = getDashboardPanelFromId(idToDuplicate);
        const childApi = children$.value[idToDuplicate];
        if (!(0, _presentationPublishing.apiHasSerializableState)(childApi)) {
          throw new Error('cannot duplicate a non-serializable panel');
        }
        const id = (0, _uuid.v4)();
        const allPanelTitles = await getPanelTitles();
        const lastTitle = (0, _presentationPublishing.apiPublishesTitle)(childApi) ? (_getTitle = (0, _presentationPublishing.getTitle)(childApi)) !== null && _getTitle !== void 0 ? _getTitle : '' : '';
        const newTitle = getClonedPanelTitle(allPanelTitles, lastTitle);

        /**
         * For embeddables that have library transforms, we need to ensure
         * to clone them with by value serialized state.
         */
        const serializedState = (0, _presentationPublishing.apiHasLibraryTransforms)(childApi) ? childApi.getSerializedStateByValue() : childApi.serializeState();
        if (serializedState.references) {
          pushReferences((0, _common.prefixReferencesFromPanel)(id, serializedState.references));
        }
        _kibana_services.coreServices.notifications.toasts.addSuccess({
          title: _dashboard_actions_strings.dashboardClonePanelActionStrings.getSuccessMessage(),
          'data-test-subj': 'addObjectToContainerSuccess'
        });
        const {
          newPanelPlacement,
          otherPanels
        } = (0, _place_clone_panel_strategy.placeClonePanel)({
          width: panelToClone.gridData.w,
          height: panelToClone.gridData.h,
          currentPanels: panels$.value,
          placeBesideId: panelToClone.explicitInput.id
        });
        const newPanel = {
          type: panelToClone.type,
          explicitInput: {
            ...serializedState.rawState,
            title: newTitle,
            id
          },
          gridData: {
            ...newPanelPlacement,
            i: id
          }
        };
        setPanels({
          ...otherPanels,
          [id]: newPanel
        });
      },
      getDashboardPanelFromId,
      getPanelCount: () => {
        return Object.keys(panels$.value).length;
      },
      getSerializedStateForChild: childId => {
        var _panels$$value$childI, _panels$$value$childI2;
        const rawState = (_panels$$value$childI = (_panels$$value$childI2 = panels$.value[childId]) === null || _panels$$value$childI2 === void 0 ? void 0 : _panels$$value$childI2.explicitInput) !== null && _panels$$value$childI !== void 0 ? _panels$$value$childI : {
          id: childId
        };
        const {
          id,
          ...serializedState
        } = rawState;
        return Object.keys(serializedState).length === 0 ? undefined : {
          rawState,
          references: getReferencesForPanelId(childId)
        };
      },
      getRuntimeStateForChild: childId => {
        var _restoredRuntimeState;
        return (_restoredRuntimeState = restoredRuntimeState) === null || _restoredRuntimeState === void 0 ? void 0 : _restoredRuntimeState[childId];
      },
      panels$,
      removePanel: id => {
        const panels = {
          ...panels$.value
        };
        if (panels[id]) {
          delete panels[id];
          setPanels(panels);
        }
        const children = {
          ...children$.value
        };
        if (children[id]) {
          delete children[id];
          children$.next(children);
        }
      },
      replacePanel: async (idToRemove, panelPackage) => {
        const panels = {
          ...panels$.value
        };
        if (!panels[idToRemove]) {
          throw new _public.PanelNotFoundError();
        }
        const id = (0, _uuid.v4)();
        const oldPanel = panels[idToRemove];
        delete panels[idToRemove];
        const {
          panelType: type,
          serializedState,
          initialState
        } = panelPackage;
        if (serializedState !== null && serializedState !== void 0 && serializedState.references && serializedState.references.length > 0) {
          pushReferences((0, _common.prefixReferencesFromPanel)(id, serializedState === null || serializedState === void 0 ? void 0 : serializedState.references));
        }
        if (initialState) setRuntimeStateForChild(id, initialState);
        setPanels({
          ...panels,
          [id]: {
            ...oldPanel,
            explicitInput: {
              ...(serializedState === null || serializedState === void 0 ? void 0 : serializedState.rawState),
              id
            },
            type
          }
        });
        const children = {
          ...children$.value
        };
        if (children[idToRemove]) {
          delete children[idToRemove];
          children$.next(children);
        }
        await untilEmbeddableLoaded(id);
        return id;
      },
      setPanels,
      setRuntimeStateForChild,
      untilEmbeddableLoaded
    },
    comparators: {
      panels: [panels$, setPanels, _are_panel_layouts_equal.arePanelLayoutsEqual]
    },
    internalApi: {
      registerChildApi: api => {
        children$.next({
          ...children$.value,
          [api.uuid]: api
        });
      },
      setPanels,
      reset: lastSavedState => {
        setPanels(lastSavedState.panels);
        restoredRuntimeState = {};
        let resetChangedPanelCount = false;
        const currentChildren = children$.value;
        for (const panelId of Object.keys(currentChildren)) {
          if (panels$.value[panelId]) {
            const child = currentChildren[panelId];
            if ((0, _presentationPublishing.apiPublishesUnsavedChanges)(child)) {
              const success = child.resetUnsavedChanges();
              if (!success) {
                _kibana_services.coreServices.notifications.toasts.addWarning(_i18n.i18n.translate('dashboard.reset.panelError', {
                  defaultMessage: 'Unable to reset panel changes'
                }));
              }
            }
          } else {
            // if reset resulted in panel removal, we need to update the list of children
            delete currentChildren[panelId];
            resetChangedPanelCount = true;
          }
        }
        if (resetChangedPanelCount) children$.next(currentChildren);
      },
      getState: () => {
        const references = [];
        const panels = Object.keys(panels$.value).reduce((acc, id) => {
          var _panels$$value$id$exp, _serializeResult$refe;
          const childApi = children$.value[id];
          const serializeResult = (0, _presentationPublishing.apiHasSerializableState)(childApi) ? childApi.serializeState() : {
            rawState: (_panels$$value$id$exp = panels$.value[id].explicitInput) !== null && _panels$$value$id$exp !== void 0 ? _panels$$value$id$exp : {},
            references: getReferencesForPanelId(id)
          };
          acc[id] = {
            ...panels$.value[id],
            explicitInput: {
              ...serializeResult.rawState,
              id
            }
          };
          references.push(...(0, _common.prefixReferencesFromPanel)(id, (_serializeResult$refe = serializeResult.references) !== null && _serializeResult$refe !== void 0 ? _serializeResult$refe : []));
          return acc;
        }, {});
        return {
          panels,
          references
        };
      }
    }
  };
}
function getClonedPanelTitle(panelTitles, rawTitle) {
  if (rawTitle === '') return '';
  const clonedTag = _dashboard_actions_strings.dashboardClonePanelActionStrings.getClonedTag();
  const cloneRegex = new RegExp(`\\(${clonedTag}\\)`, 'g');
  const cloneNumberRegex = new RegExp(`\\(${clonedTag} [0-9]+\\)`, 'g');
  const baseTitle = rawTitle.replace(cloneNumberRegex, '').replace(cloneRegex, '').trim();
  const similarTitles = (0, _lodash.filter)(panelTitles, title => {
    return title.startsWith(baseTitle);
  });
  const cloneNumbers = (0, _lodash.map)(similarTitles, title => {
    if (title.match(cloneRegex)) return 0;
    const cloneTag = title.match(cloneNumberRegex);
    return cloneTag ? parseInt(cloneTag[0].replace(/[^0-9.]/g, ''), 10) : -1;
  });
  const similarBaseTitlesCount = (0, _lodash.max)(cloneNumbers) || 0;
  return similarBaseTitlesCount < 0 ? baseTitle + ` (${clonedTag})` : baseTitle + ` (${clonedTag} ${similarBaseTitlesCount + 1})`;
}