"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BaseUiSettingsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _ui_settings_errors = require("../ui_settings_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Base implementation of the {@link IUiSettingsClient}.
 */
class BaseUiSettingsClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "defaults", void 0);
    (0, _defineProperty2.default)(this, "overrides", void 0);
    (0, _defineProperty2.default)(this, "log", void 0);
    const {
      defaults = {},
      overrides = {},
      log
    } = options;
    this.log = log;
    this.overrides = overrides;
    this.defaults = defaults;
  }
  getRegistered() {
    const copiedDefaults = {};
    for (const [key, value] of Object.entries(this.defaults)) {
      copiedDefaults[key] = (0, _lodash.omit)(value, 'schema');
    }
    return copiedDefaults;
  }
  async get(key, context) {
    const all = await this.getAll(context);
    return all[key];
  }
  async getAll(context) {
    const defaultValues = await this.getDefaultValues(context);
    const result = {
      ...defaultValues
    };
    const userProvided = await this.getUserProvided();
    Object.keys(userProvided).forEach(key => {
      if (userProvided[key].userValue !== undefined) {
        result[key] = userProvided[key].userValue;
      }
    });
    return Object.freeze(result);
  }
  isOverridden(key) {
    return Object.hasOwn(this.overrides, key);
  }
  isSensitive(key) {
    const definition = this.defaults[key];
    return !!(definition !== null && definition !== void 0 && definition.sensitive);
  }
  async validate(key, value) {
    if (value == null) {
      throw new _ui_settings_errors.ValidationBadValueError();
    }
    const definition = this.defaults[key];
    if (!definition) {
      throw new _ui_settings_errors.ValidationSettingNotFoundError(key);
    }
    if (definition.schema) {
      try {
        definition.schema.validate(value);
      } catch (error) {
        return {
          valid: false,
          errorMessage: error.message
        };
      }
    }
    return {
      valid: true
    };
  }
  validateKey(key, value) {
    const definition = this.defaults[key];
    if (value === null || definition === undefined) return;
    if (definition.schema) {
      definition.schema.validate(value, {}, `validation [${key}]`);
    }
  }
  async getDefaultValues(context) {
    const values = {};
    const promises = [];
    for (const [key, definition] of Object.entries(this.defaults)) {
      if (definition.getValue) {
        promises.push([key, definition.getValue(context)]);
      } else {
        values[key] = definition.value;
      }
    }
    await Promise.all(promises.map(([key, promise]) => promise.then(value => {
      values[key] = value;
    }).catch(error => {
      this.log.error(`[UiSettingsClient] Failed to get value for key "${key}": ${error}`);
      // Fallback to `value` prop if `getValue()` fails
      values[key] = this.defaults[key].value;
    })));
    return values;
  }
}
exports.BaseUiSettingsClient = BaseUiSettingsClient;