"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerImportRoute = void 0;
var _path = require("path");
var _configSchema = require("@kbn/config-schema");
var _coreSavedObjectsImportExportServerInternal = require("@kbn/core-saved-objects-import-export-server-internal");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerImportRoute = (router, {
  config,
  coreUsageData
}) => {
  const {
    maxImportPayloadBytes
  } = config;
  router.post({
    path: '/_import',
    options: {
      summary: `Import saved objects`,
      tags: ['oas-tag:saved objects'],
      access: 'public',
      description: 'Create sets of Kibana saved objects from a file created by the export API. Saved objects can only be imported into the same version, a newer minor on the same major, or the next major. Exported saved objects are not backwards compatible and cannot be imported into an older version of Kibana.',
      body: {
        maxBytes: maxImportPayloadBytes,
        output: 'stream',
        accepts: 'multipart/form-data'
      }
    },
    validate: {
      query: _configSchema.schema.object({
        overwrite: _configSchema.schema.boolean({
          defaultValue: false
        }),
        createNewCopies: _configSchema.schema.boolean({
          defaultValue: false
        }),
        compatibilityMode: _configSchema.schema.boolean({
          defaultValue: false
        })
      }, {
        validate: object => {
          if (object.overwrite && object.createNewCopies) {
            return 'cannot use [overwrite] with [createNewCopies]';
          }
          if (object.createNewCopies && object.compatibilityMode) {
            return 'cannot use [createNewCopies] with [compatibilityMode]';
          }
        }
      }),
      body: _configSchema.schema.object({
        file: _configSchema.schema.stream()
      })
    }
  }, (0, _utils.catchAndReturnBoomErrors)(async (context, request, response) => {
    const {
      overwrite,
      createNewCopies,
      compatibilityMode
    } = request.query;
    const {
      getClient,
      getImporter,
      typeRegistry
    } = (await context.core).savedObjects;
    const usageStatsClient = coreUsageData.getClient();
    usageStatsClient.incrementSavedObjectsImport({
      request,
      createNewCopies,
      overwrite,
      compatibilityMode
    }).catch(() => {});
    const file = request.body.file;
    const fileExtension = (0, _path.extname)(file.hapi.filename).toLowerCase();
    if (fileExtension !== '.ndjson') {
      return response.badRequest({
        body: `Invalid file extension ${fileExtension}`
      });
    }
    let readStream;
    try {
      readStream = await (0, _utils.createSavedObjectsStreamFromNdJson)(file);
    } catch (e) {
      return response.badRequest({
        body: e
      });
    }
    const supportedTypes = typeRegistry.getImportableAndExportableTypes().map(t => t.name);
    const includedHiddenTypes = supportedTypes.filter(supportedType => typeRegistry.isHidden(supportedType));
    const client = getClient({
      includedHiddenTypes
    });
    const importer = getImporter(client);
    try {
      const result = await importer.import({
        readStream,
        overwrite,
        createNewCopies,
        compatibilityMode
      });
      return response.ok({
        body: result
      });
    } catch (e) {
      if (e instanceof _coreSavedObjectsImportExportServerInternal.SavedObjectsImportError) {
        return response.badRequest({
          body: {
            message: e.message,
            attributes: e.attributes
          }
        });
      }
      throw e;
    }
  }));
};
exports.registerImportRoute = registerImportRoute;