"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Router = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _nodeEvents = require("node:events");
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _esErrors = require("@kbn/es-errors");
var _versioned_router = require("./versioned_router");
var _request = require("./request");
var _response = require("./response");
var _response_adapter = require("./response_adapter");
var _error_wrapper = require("./error_wrapper");
var _util = require("./util");
var _strip_illegal_http2_headers = require("./strip_illegal_http2_headers");
var _route = require("./route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** @internal */

/**
 * We have at least two implementations of InternalRouterRoutes:
 * (1) Router route
 * (2) Versioned router route {@link CoreVersionedRoute}
 *
 * The former registers internal handlers when users call `route.put(...)` while
 * the latter registers an internal handler for `router.versioned.put(...)`.
 *
 * This enables us to expose internal details to each of these types routes so
 * that implementation has freedom to change what it needs to in each case, like:
 *
 * validation: versioned routes only know what validation to run after inspecting
 * special version values, whereas "regular" routes only ever have one validation
 * that is predetermined to always run.
 * @internal
 */

/** @internal */

/** @internal */

/** @internal */

/**
 * @internal
 */
class Router {
  constructor(routerPath, log, enhanceWithContext, options) {
    (0, _defineProperty2.default)(this, "routes", []);
    (0, _defineProperty2.default)(this, "pluginId", void 0);
    (0, _defineProperty2.default)(this, "get", void 0);
    (0, _defineProperty2.default)(this, "post", void 0);
    (0, _defineProperty2.default)(this, "delete", void 0);
    (0, _defineProperty2.default)(this, "put", void 0);
    (0, _defineProperty2.default)(this, "patch", void 0);
    (0, _defineProperty2.default)(this, "handleLegacyErrors", _error_wrapper.wrapErrors);
    (0, _defineProperty2.default)(this, "emitPostValidate", (request, postValidateConext = {
      isInternalApiRequest: true,
      isPublicAccess: false
    }) => {
      const postValidate = 'onPostValidate';
      Router.events.emit(postValidate, request, postValidateConext);
    });
    (0, _defineProperty2.default)(this, "versionedRouter", undefined);
    this.routerPath = routerPath;
    this.log = log;
    this.enhanceWithContext = enhanceWithContext;
    this.options = options;
    this.pluginId = options.pluginId;
    const buildMethod = method => (route, handler) => {
      this.registerRoute((0, _route.buildRoute)({
        handler: this.enhanceWithContext(handler),
        log: this.log,
        method,
        route,
        router: this
      }));
    };
    this.get = buildMethod('get');
    this.post = buildMethod('post');
    this.delete = buildMethod('delete');
    this.put = buildMethod('put');
    this.patch = buildMethod('patch');
  }
  static on(event, cb) {
    Router.events.on(event, cb);
  }
  static off(event, cb) {
    Router.events.off(event, cb);
  }
  getRoutes({
    excludeVersionedRoutes
  } = {}) {
    if (excludeVersionedRoutes) {
      return this.routes.filter(route => !route.isVersioned);
    }
    return [...this.routes];
  }
  /** @internal */
  registerRoute(route) {
    this.routes.push({
      ...route,
      handler: async (request, responseToolkit) => await this.handle({
        request,
        responseToolkit,
        handler: route.handler
      })
    });
  }
  async handle({
    request,
    responseToolkit,
    handler
  }) {
    const hapiResponseAdapter = new _response_adapter.HapiResponseAdapter(responseToolkit);
    try {
      const kibanaResponse = await handler(request);
      if ((0, _request.getProtocolFromRequest)(request) === 'http2' && kibanaResponse.options.headers) {
        var _this$options$isDev;
        kibanaResponse.options.headers = (0, _strip_illegal_http2_headers.stripIllegalHttp2Headers)({
          headers: kibanaResponse.options.headers,
          isDev: (_this$options$isDev = this.options.isDev) !== null && _this$options$isDev !== void 0 ? _this$options$isDev : false,
          logger: this.log,
          requestContext: `${request.route.method} ${request.route.path}`
        });
      }
      return hapiResponseAdapter.handle(kibanaResponse);
    } catch (error) {
      // capture error
      _elasticApmNode.default.captureError(error);

      // forward 401 errors from ES client
      if ((0, _esErrors.isUnauthorizedError)(error)) {
        this.log.error('401 Unauthorized', (0, _util.formatErrorMeta)(401, {
          request,
          error
        }));
        return hapiResponseAdapter.handle(_response.kibanaResponseFactory.unauthorized(convertEsUnauthorized(error)));
      }

      // return a generic 500 to avoid error info / stack trace surfacing
      this.log.error('500 Server Error', (0, _util.formatErrorMeta)(500, {
        request,
        error
      }));
      return hapiResponseAdapter.toInternalError();
    }
  }
  get versioned() {
    if (this.versionedRouter === undefined) {
      this.versionedRouter = _versioned_router.CoreVersionedRouter.from({
        router: this,
        isDev: this.options.isDev,
        log: this.log,
        ...this.options.versionedRouterOptions
      });
    }
    return this.versionedRouter;
  }
}
exports.Router = Router;
/**
 * Used for global request events at the router level, similar to what we get from Hapi's request lifecycle events.
 *
 * See {@link RouterEvents}.
 */
(0, _defineProperty2.default)(Router, "events", new _nodeEvents.EventEmitter());
const convertEsUnauthorized = e => {
  const getAuthenticateHeaderValue = () => {
    const header = Object.entries(e.headers || {}).find(([key]) => key.toLowerCase() === 'www-authenticate');
    return header ? header[1] : 'Basic realm="Authorization Required"';
  };
  return {
    body: e.message,
    headers: {
      'www-authenticate': getAuthenticateHeaderValue()
    }
  };
};