"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FatalErrorsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _react = _interopRequireDefault(require("react"));
var _reactDom = require("react-dom");
var _rxjs = require("rxjs");
var _reactKibanaContextRoot = require("@kbn/react-kibana-context-root");
var _fatal_errors_screen = require("./fatal_errors_screen");
var _get_error_info = require("./get_error_info");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** @internal */

/** @internal */
class FatalErrorsService {
  /**
   *
   * @param rootDomElement
   * @param onFirstErrorCb - Callback function that gets executed after the first error,
   *   but before the FatalErrorsService renders the error to the DOM.
   */
  constructor(rootDomElement, onFirstErrorCb) {
    (0, _defineProperty2.default)(this, "errorInfo$", new _rxjs.ReplaySubject());
    (0, _defineProperty2.default)(this, "fatalErrors", void 0);
    this.rootDomElement = rootDomElement;
    this.onFirstErrorCb = onFirstErrorCb;
  }
  setup(deps) {
    this.errorInfo$.pipe((0, _rxjs.first)(), (0, _rxjs.tap)(() => {
      this.onFirstErrorCb();
      this.renderError(deps);
    })).subscribe({
      error: error => {
        // eslint-disable-next-line no-console
        console.error('Uncaught error in fatal error service internals', error);
      }
    });
    this.fatalErrors = {
      add: (error, source) => {
        const errorInfo = (0, _get_error_info.getErrorInfo)(error, source);
        this.errorInfo$.next(errorInfo);
        if (error instanceof Error) {
          // make stack traces clickable by putting whole error in the console
          // eslint-disable-next-line no-console
          console.error(error);
        }
        throw error;
      },
      get$: () => {
        return this.errorInfo$.asObservable();
      }
    };
    this.setupGlobalErrorHandlers();
    return this.fatalErrors;
  }
  start() {
    const {
      fatalErrors
    } = this;
    if (!fatalErrors) {
      throw new Error('FatalErrorsService#setup() must be invoked before start.');
    }
    return fatalErrors;
  }
  renderError({
    analytics,
    i18n,
    theme,
    injectedMetadata
  }) {
    // delete all content in the rootDomElement
    this.rootDomElement.textContent = '';

    // create and mount a container for the <FatalErrorScreen>
    const container = document.createElement('div');
    this.rootDomElement.appendChild(container);
    (0, _reactDom.render)( /*#__PURE__*/_react.default.createElement(_reactKibanaContextRoot.KibanaRootContextProvider, {
      analytics: analytics,
      i18n: i18n,
      theme: theme,
      globalStyles: true
    }, /*#__PURE__*/_react.default.createElement(_fatal_errors_screen.FatalErrorsScreen, {
      buildNumber: injectedMetadata.getKibanaBuildNumber(),
      kibanaVersion: injectedMetadata.getKibanaVersion(),
      errorInfo$: this.errorInfo$
    })), container);
  }
  setupGlobalErrorHandlers() {
    if (window.addEventListener) {
      window.addEventListener('unhandledrejection', e => {
        const {
          message,
          stack
        } = (0, _get_error_info.getErrorInfo)(e.reason);
        // eslint-disable-next-line no-console
        console.log(`Detected an unhandled Promise rejection.\n
        Message: ${message}\n
        Stack: ${stack}`);
      });
    }
  }
}
exports.FatalErrorsService = FatalErrorsService;