"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.logCoreStatusChanges = void 0;
var _rxjs = require("rxjs");
var _coreStatusCommon = require("@kbn/core-status-common");
var _log_throttled_buffer = require("./log_throttled_buffer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// let services log up to 3 status changes every 30s (extra messages will be throttled / aggregated)
const MAX_MESSAGES_PER_SERVICE_PER_INTERVAL = 3;
const THROTTLE_INTERVAL_MILLIS = 30000;
const MAX_THROTTLED_MESSAGES = 10;
const logCoreStatusChanges = ({
  logger,
  core$,
  stop$,
  maxMessagesPerServicePerInterval = MAX_MESSAGES_PER_SERVICE_PER_INTERVAL,
  throttleIntervalMillis = THROTTLE_INTERVAL_MILLIS,
  maxThrottledMessages = MAX_THROTTLED_MESSAGES
}) => {
  const buffer$ = new _rxjs.Subject();
  const throttled$ = (0, _log_throttled_buffer.createLogThrottledBuffer)({
    buffer$,
    stop$,
    maxThrottledMessages
  });
  const lastMessagesTimestamps = {};
  const direct$ = core$.pipe((0, _rxjs.startWith)(undefined),
  // consider all services unavailable by default
  (0, _rxjs.takeUntil)(stop$), (0, _rxjs.pairwise)(), (0, _rxjs.map)(([previous, current]) => getServiceUpdates({
    previous,
    current: current
  })), (0, _rxjs.concatAll)(), (0, _rxjs.filter)(serviceStatus => {
    const now = Date.now();
    const pluginQuota = lastMessagesTimestamps[serviceStatus.name] || [];
    lastMessagesTimestamps[serviceStatus.name] = pluginQuota;

    // remove timestamps of messages older than the threshold
    while (pluginQuota.length > 0 && pluginQuota[0] < now - throttleIntervalMillis) {
      pluginQuota.shift();
    }
    if (pluginQuota.length >= maxMessagesPerServicePerInterval) {
      // we're still over quota, throttle the message
      buffer$.next(serviceStatus);
      return false;
    } else {
      // let the message pass through
      pluginQuota.push(now);
      return true;
    }
  }));
  return (0, _rxjs.merge)(direct$, throttled$).subscribe(event => {
    if (typeof event === 'string') {
      logger.warn(event);
    } else {
      const serviceStatus = event;
      const {
        name
      } = serviceStatus;
      const serviceLogger = logger.get(name);
      const message = getServiceStatusMessage(serviceStatus);
      switch (serviceStatus.level) {
        case _coreStatusCommon.ServiceStatusLevels.available:
          serviceLogger.info(message);
          break;
        case _coreStatusCommon.ServiceStatusLevels.degraded:
          serviceLogger.warn(message);
          break;
        default:
          serviceLogger.error(message);
      }
    }
  });
};
exports.logCoreStatusChanges = logCoreStatusChanges;
const getServiceUpdates = ({
  current,
  previous
}) => {
  let name;
  const updated = [];
  for (name in current) {
    if (Object.hasOwn(current, name)) {
      const currentLevel = current[name].level;
      const previousLevel = previous === null || previous === void 0 ? void 0 : previous[name].level;
      if (currentLevel !== previousLevel) {
        updated.push({
          ...current[name],
          name
        });
      }
    }
  }
  return updated;
};
const getServiceStatusMessage = ({
  name,
  level,
  summary,
  detail,
  repeats = 0
}) => `${name} service is now ${level === null || level === void 0 ? void 0 : level.toString()}: ${summary}${detail ? ` | ${detail}` : ''}${repeats > 1 ? ` (repeated ${repeats} times)` : ''}`;