"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSummaryStatus = void 0;
var _coreStatusCommon = require("@kbn/core-status-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns a single {@link ServiceStatus} that summarizes the most severe status level from a group of statuses.
 */
const getSummaryStatus = ({
  serviceStatuses,
  pluginStatuses
}) => {
  const {
    highestLevel,
    highestLevelServices,
    highestLevelPlugins
  } = highestLevelSummary({
    serviceStatuses,
    pluginStatuses
  });
  if (highestLevel === _coreStatusCommon.ServiceStatusLevels.available) {
    return {
      level: _coreStatusCommon.ServiceStatusLevels.available,
      summary: serviceStatuses && pluginStatuses ? 'All services and plugins are available' : serviceStatuses ? 'All services are available' : 'All plugins are available'
    };
  } else {
    const failingPlugins = highestLevelPlugins === null || highestLevelPlugins === void 0 ? void 0 : highestLevelPlugins.filter(({
      reported
    }) => reported);
    const affectedPlugins = highestLevelPlugins === null || highestLevelPlugins === void 0 ? void 0 : highestLevelPlugins.filter(({
      reported
    }) => !reported);
    const failingServicesNames = highestLevelServices === null || highestLevelServices === void 0 ? void 0 : highestLevelServices.map(({
      name
    }) => name);
    const failingPluginsNames = failingPlugins === null || failingPlugins === void 0 ? void 0 : failingPlugins.map(({
      name
    }) => name);
    const affectedPluginsNames = affectedPlugins === null || affectedPlugins === void 0 ? void 0 : affectedPlugins.map(({
      name
    }) => name);
    return {
      level: highestLevel,
      summary: getSummaryContent({
        level: highestLevel,
        services: failingServicesNames,
        plugins: failingPluginsNames
      }),
      // TODO: include URL to status page
      detail: `See the status page for more information`,
      meta: {
        failingServices: failingServicesNames,
        failingPlugins: failingPluginsNames,
        affectedPlugins: affectedPluginsNames
      }
    };
  }
};
exports.getSummaryStatus = getSummaryStatus;
const getSummaryContent = ({
  level,
  services,
  plugins
}) => {
  const list = [...services, ...plugins].join(', ');
  return `${services.length} service(s) and ${plugins.length} plugin(s) are ${level.toString()}: ${list}`;
};
const highestLevelSummary = ({
  serviceStatuses,
  pluginStatuses
}) => {
  let highestServiceLevel = _coreStatusCommon.ServiceStatusLevels.available;
  let highestPluginLevel = _coreStatusCommon.ServiceStatusLevels.available;
  let highestLevelServices = [];
  let highestLevelPlugins = [];
  if (serviceStatuses) {
    let name;
    for (name in serviceStatuses) {
      if (Object.hasOwn(serviceStatuses, name)) {
        const namedStatus = {
          ...serviceStatuses[name],
          name
        };
        if (serviceStatuses[name].level === highestServiceLevel) {
          highestLevelServices.push(namedStatus);
        } else if (serviceStatuses[name].level > highestServiceLevel) {
          highestLevelServices = [namedStatus];
          highestServiceLevel = serviceStatuses[name].level;
        }
      }
    }
  }
  if (pluginStatuses) {
    Object.entries(pluginStatuses).forEach(([name, pluginStatus]) => {
      const namedStatus = {
        ...pluginStatus,
        name
      };
      if (pluginStatus.level === highestPluginLevel) {
        highestLevelPlugins.push(namedStatus);
      } else if (pluginStatus.level > highestPluginLevel) {
        highestLevelPlugins = [namedStatus];
        highestPluginLevel = pluginStatus.level;
      }
    });
  }
  if (highestServiceLevel === highestPluginLevel) {
    return {
      highestLevel: highestServiceLevel,
      highestLevelServices,
      highestLevelPlugins
    };
  } else if (highestServiceLevel > highestPluginLevel) {
    return {
      highestLevel: highestServiceLevel,
      highestLevelServices,
      highestLevelPlugins: []
    };
  } else {
    return {
      highestLevel: highestPluginLevel,
      highestLevelServices: [],
      highestLevelPlugins
    };
  }
};