"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerResolveRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerResolveRoute = (router, {
  config,
  coreUsageData,
  logger,
  access,
  deprecationInfo
}) => {
  const {
    allowHttpApiAccess
  } = config;
  router.get({
    path: '/resolve/{type}/{id}',
    options: {
      summary: `Resolve a saved object`,
      tags: ['oas-tag:saved objects'],
      access,
      deprecated: deprecationInfo,
      description: `Retrieve a single Kibana saved object by ID, using any legacy URL alias if it exists.
        Under certain circumstances, when Kibana is upgraded, saved object migrations may necessitate regenerating some object IDs to enable new features. When an object's ID is regenerated, a legacy URL alias is created for that object, preserving its old ID. In such a scenario, that object can be retrieved with the resolve API using either its new ID or its old ID.`
    },
    validate: {
      params: _configSchema.schema.object({
        type: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      })
    }
  }, router.handleLegacyErrors(async (context, request, response) => {
    (0, _utils.logWarnOnExternalRequest)({
      method: 'get',
      path: '/api/saved_objects/resolve/{type}/{id}',
      request,
      logger
    });
    const {
      type,
      id
    } = request.params;
    const {
      savedObjects
    } = await context.core;
    const usageStatsClient = coreUsageData.getClient();
    usageStatsClient.incrementSavedObjectsResolve({
      request,
      types: [type]
    }).catch(() => {});
    if (!allowHttpApiAccess) {
      (0, _utils.throwIfTypeNotVisibleByAPI)(type, savedObjects.typeRegistry);
    }
    const result = await savedObjects.client.resolve(type, id, {
      migrationVersionCompatibility: 'compatible'
    });
    return response.ok({
      body: result
    });
  }));
};
exports.registerResolveRoute = registerResolveRoute;