"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineBulkActionCspBenchmarkRulesRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _v = require("@kbn/cloud-security-posture-common/schema/rules/v4");
var _constants = require("../../../../common/constants");
var _v2 = require("./v1");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
	This API allows bulk actions (mute or unmute) on CSP benchmark rules.
	Request:
	{
	  action: 'mute' | 'unmute'; // Specify the bulk action type (mute or unmute)
	  rules: [
	    {
        benchmark_id: string;       // Identifier for the CSP benchmark
	      benchmark_version: string;  // Version of the CSP benchmark
	      rule_number: string;        // Rule number within the benchmark
	      rule_id: string;            // Unique identifier for the rule
	    },
	    // ... (additional benchmark rules)
	  ];
	}

	Response:
	{
	  updated_benchmark_rules: CspBenchmarkRulesStates; Benchmark rules object that were affected
	  detection_rules: string;         // Status message indicating the number of detection rules affected
	  message: string;                 // Success message
	}
	*/
const defineBulkActionCspBenchmarkRulesRoute = router => router.versioned.post({
  access: 'internal',
  path: _constants.CSP_BENCHMARK_RULES_BULK_ACTION_ROUTE_PATH,
  security: {
    authz: {
      requiredPrivileges: ['cloud-security-posture-all']
    }
  }
}).addVersion({
  version: '1',
  validate: {
    request: {
      body: _v.cspBenchmarkRulesBulkActionRequestSchema
    }
  }
}, async (context, request, response) => {
  const cspContext = await context.csp;
  try {
    const requestBody = request.body;
    const benchmarkRulesToUpdate = requestBody.rules;
    const detectionRulesClient = await (await context.alerting).getRulesClient();
    const handlerResponse = await (0, _v2.bulkActionBenchmarkRulesHandler)(cspContext.soClient, cspContext.encryptedSavedObjects, detectionRulesClient, benchmarkRulesToUpdate, requestBody.action, cspContext.logger);
    const body = {
      updated_benchmark_rules: handlerResponse.updatedBenchmarkRulesStates,
      message: 'The bulk operation has been executed successfully.'
    };
    if (requestBody.action === 'mute' && handlerResponse.disabledDetectionRules) {
      body.disabled_detection_rules = handlerResponse.disabledDetectionRules;
    }
    return response.ok({
      body
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    cspContext.logger.error(`Bulk action failed: ${error.message}`);
    return response.customError({
      body: {
        message: error.message
      },
      statusCode: error.statusCode || 500 // Default to 500 if no specific status code is provided
    });
  }
});
exports.defineBulkActionCspBenchmarkRulesRoute = defineBulkActionCspBenchmarkRulesRoute;