"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAccountsStats = void 0;
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAccountsStatsQuery = () => ({
  index: _constants.LOGS_CLOUD_DEFEND_PATTERN,
  query: {
    match_all: {}
  },
  aggs: {
    accounts: {
      terms: {
        field: 'orchestrator.cluster.id',
        order: {
          _count: 'desc'
        },
        size: 100
      },
      aggs: {
        nodes_count: {
          cardinality: {
            field: 'cloud.instance.name'
          }
        },
        agents_count: {
          cardinality: {
            field: 'agent.id'
          }
        },
        kubernetes_version: {
          top_metrics: {
            metrics: {
              field: 'orchestrator.version'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        cloud_provider: {
          top_metrics: {
            metrics: {
              field: 'cloud.provider'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        file_doc_count: {
          filter: {
            bool: {
              filter: [{
                bool: {
                  should: [{
                    term: {
                      'event.category': 'file'
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          }
        },
        process_doc_count: {
          filter: {
            bool: {
              filter: [{
                bool: {
                  should: [{
                    term: {
                      'event.category': 'process'
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          }
        },
        alert_doc_count: {
          filter: {
            bool: {
              filter: [{
                bool: {
                  should: [{
                    term: {
                      'event.kind': 'alert'
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          }
        },
        pods_count: {
          cardinality: {
            field: 'orchestrator.resource.name'
          }
        }
      }
    }
  },
  size: 0,
  _source: false
});
const getCloudDefendAccountsStats = (aggregatedResourcesStats, logger) => {
  const accounts = aggregatedResourcesStats.accounts.buckets;
  const cloudDefendAccountsStats = accounts.map(account => {
    var _account$kubernetes_v, _account$kubernetes_v2, _account$kubernetes_v3, _account$cloud_provid, _account$cloud_provid2, _account$cloud_provid3;
    return {
      account_id: account.key,
      total_doc_count: account.doc_count,
      file_doc_count: account.file_doc_count.doc_count,
      process_doc_count: account.process_doc_count.doc_count,
      alert_doc_count: account.alert_doc_count.doc_count,
      kubernetes_version: (_account$kubernetes_v = account.kubernetes_version) === null || _account$kubernetes_v === void 0 ? void 0 : (_account$kubernetes_v2 = _account$kubernetes_v.top) === null || _account$kubernetes_v2 === void 0 ? void 0 : (_account$kubernetes_v3 = _account$kubernetes_v2[0]) === null || _account$kubernetes_v3 === void 0 ? void 0 : _account$kubernetes_v3.metrics['orchestrator.version'],
      cloud_provider: (_account$cloud_provid = account.cloud_provider) === null || _account$cloud_provid === void 0 ? void 0 : (_account$cloud_provid2 = _account$cloud_provid.top) === null || _account$cloud_provid2 === void 0 ? void 0 : (_account$cloud_provid3 = _account$cloud_provid2[0]) === null || _account$cloud_provid3 === void 0 ? void 0 : _account$cloud_provid3.metrics['cloud.provider'],
      agents_count: account.agents_count.value,
      nodes_count: account.nodes_count.value,
      pods_count: account.pods_count.value
    };
  });
  logger.info('CloudDefend telemetry: accounts stats was sent');
  return cloudDefendAccountsStats;
};
const getAccountsStats = async (esClient, logger) => {
  try {
    const isIndexExists = await esClient.indices.exists({
      index: _constants.LOGS_CLOUD_DEFEND_PATTERN
    });
    if (isIndexExists) {
      const accountsStatsResponse = await esClient.search(getAccountsStatsQuery());
      const cloudDefendAccountsStats = accountsStatsResponse.aggregations ? getCloudDefendAccountsStats(accountsStatsResponse.aggregations, logger) : [];
      return cloudDefendAccountsStats;
    }
    return [];
  } catch (e) {
    logger.error(`Failed to get account stats ${e}`);
    return [];
  }
};
exports.getAccountsStats = getAccountsStats;