"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseUrlParams = parseUrlParams;
var _rison = require("@kbn/rison");
var _lodash = require("lodash");
var _constants = require("../../../containers/constants");
var _ = require(".");
var _ui = require("../../../../common/ui");
var _constants2 = require("../constants");
var _schema = require("../schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const legacyDefaultState = {
  page: 1,
  perPage: 10,
  sortField: _ui.SortFieldCase.createdAt,
  sortOrder: 'desc',
  status: [],
  severity: []
};

/**
 * Parses legacy state in URL.
 *
 * - Parameters in the query string can have multiple formats:
 *   1. Comma-separated values (e.g., "status=foo,bar")
 *   2. A single value (e.g., "status=foo")
 *   3. Repeated keys (e.g., "status=foo&status=bar")
 *
 */
const parseLegacyUrl = urlParams => {
  const urlParamsMap = new Map();
  urlParams.forEach((value, key) => {
    if (_constants2.LEGACY_SUPPORTED_STATE_KEYS.includes(key)) {
      var _urlParamsMap$get;
      const values = (_urlParamsMap$get = urlParamsMap.get(key)) !== null && _urlParamsMap$get !== void 0 ? _urlParamsMap$get : new Set();
      value.split(',').filter(Boolean).forEach(urlValue => values.add(urlValue));
      urlParamsMap.set(key, values);
    }
  });
  const entries = new Map([...urlParamsMap].map(([key, value]) => [key, parseValue(value, legacyDefaultState[key])]));
  const params = Object.fromEntries(entries.entries());
  const allCasesParams = {
    ...params
  };
  if (params.page) {
    allCasesParams.page = (0, _.stringToIntegerWithDefault)(Array.isArray(params.page) ? params.page[0] : params.page, _constants.DEFAULT_CASES_TABLE_STATE.queryParams.page);
  }
  if (params.perPage) {
    allCasesParams.perPage = (0, _.stringToIntegerWithDefault)(Array.isArray(params.perPage) ? params.perPage[0] : params.perPage, _constants.DEFAULT_CASES_TABLE_STATE.queryParams.perPage);
  }
  if (params.status) {
    const statusAsArray = Array.isArray(params.status) ? params.status : [params.status];
    allCasesParams.status = statusAsArray.filter(notAll).filter(Boolean);
  }
  if (params.severity) {
    const severityAsArray = Array.isArray(params.severity) ? params.severity : [params.severity];
    allCasesParams.severity = severityAsArray.filter(notAll).filter(Boolean);
  }
  return allCasesParams;
};
const parseValue = (values, defaultValue) => {
  var _valuesAsArray$;
  const valuesAsArray = Array.from(values.values());
  return Array.isArray(defaultValue) ? valuesAsArray : (_valuesAsArray$ = valuesAsArray[0]) !== null && _valuesAsArray$ !== void 0 ? _valuesAsArray$ : '';
};
const notAll = option => option !== 'all';
function parseUrlParams(urlParams) {
  const allCasesParams = urlParams.get(_constants2.ALL_CASES_STATE_URL_KEY);
  if (!allCasesParams) {
    return parseAndValidateLegacyUrl(urlParams);
  }
  const parsedAllCasesParams = (0, _rison.safeDecode)(allCasesParams);
  if (!parsedAllCasesParams || !(0, _lodash.isPlainObject)(parsedAllCasesParams)) {
    return {};
  }
  const validatedAllCasesParams = (0, _schema.validateSchema)(parsedAllCasesParams, _schema.AllCasesURLQueryParamsRt);
  if (!validatedAllCasesParams) {
    return {};
  }
  return validatedAllCasesParams;
}
const parseAndValidateLegacyUrl = urlParams => {
  const validatedUrlParams = (0, _schema.validateSchema)(parseLegacyUrl(urlParams), _schema.AllCasesURLQueryParamsRt);
  if (!validatedUrlParams) {
    return {};
  }
  return validatedUrlParams;
};