"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cancelEsRequestOnAbort = cancelEsRequestOnAbort;
exports.createEntitiesESClient = createEntitiesESClient;
var _entitiesSchema = require("@kbn/entities-schema");
var _unwrap_es_response = require("@kbn/observability-plugin/common/utils/unwrap_es_response");
var _with_apm_span = require("../../../../utils/with_apm_span");
var _types = require("../../../../routes/entities/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SERVICE_ENTITIES_LATEST_ALIAS = (0, _entitiesSchema.entitiesAliasPattern)({
  type: _types.EntityType.SERVICE,
  dataset: _entitiesSchema.ENTITY_LATEST
});
const SERVICE_ENTITIES_HISTORY_ALIAS = (0, _entitiesSchema.entitiesAliasPattern)({
  type: _types.EntityType.SERVICE,
  dataset: _entitiesSchema.ENTITY_HISTORY
});
function cancelEsRequestOnAbort(promise, request, controller) {
  const subscription = request.events.aborted$.subscribe(() => {
    controller.abort();
  });
  return promise.finally(() => subscription.unsubscribe());
}
async function createEntitiesESClient({
  request,
  esClient
}) {
  function search(indexName, operationName, searchRequest) {
    const controller = new AbortController();
    const promise = (0, _with_apm_span.withApmSpan)(operationName, () => {
      return cancelEsRequestOnAbort(esClient.search({
        ...searchRequest,
        index: [indexName],
        ignore_unavailable: true
      }, {
        signal: controller.signal,
        meta: true
      }), request, controller);
    });
    return (0, _unwrap_es_response.unwrapEsResponse)(promise);
  }
  return {
    searchLatest(operationName, searchRequest) {
      return search(SERVICE_ENTITIES_LATEST_ALIAS, operationName, searchRequest);
    },
    searchHistory(operationName, searchRequest) {
      return search(SERVICE_ENTITIES_HISTORY_ALIAS, operationName, searchRequest);
    },
    async msearch(allSearches) {
      const searches = allSearches.map(params => {
        const searchParams = [{
          index: [SERVICE_ENTITIES_LATEST_ALIAS],
          ignore_unavailable: true
        }, {
          ...params.body
        }];
        return searchParams;
      }).flat();
      const promise = esClient.msearch({
        searches
      }, {
        meta: true
      });
      const {
        body
      } = await promise;
      return {
        responses: body.responses
      };
    }
  };
}