"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LegacyAlertsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _alert = require("../alert/alert");
var _create_alert_factory = require("../alert/create_alert_factory");
var _lib = require("../lib");
var _trim_recovered_alerts = require("../lib/trim_recovered_alerts");
var _log_alerts = require("../task_runner/log_alerts");
var _rules_settings = require("../../common/rules_settings");
var _config = require("../config");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class LegacyAlertsClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "maxAlerts", _config.DEFAULT_MAX_ALERTS);
    (0, _defineProperty2.default)(this, "flappingSettings", _rules_settings.DEFAULT_FLAPPING_SETTINGS);
    (0, _defineProperty2.default)(this, "ruleLogPrefix", '');
    (0, _defineProperty2.default)(this, "startedAtString", null);
    // Alerts from the previous execution that are deserialized from the task state
    (0, _defineProperty2.default)(this, "trackedAlerts", {
      active: {},
      recovered: {}
    });
    // Alerts reported from the rule executor using the alert factory
    (0, _defineProperty2.default)(this, "reportedAlerts", {});
    (0, _defineProperty2.default)(this, "processedAlerts", void 0);
    (0, _defineProperty2.default)(this, "alertFactory", void 0);
    this.options = options;
    this.processedAlerts = {
      new: {},
      active: {},
      activeCurrent: {},
      recovered: {},
      recoveredCurrent: {}
    };
  }
  async initializeExecution({
    maxAlerts,
    ruleLabel,
    startedAt,
    flappingSettings,
    activeAlertsFromState,
    recoveredAlertsFromState
  }) {
    var _this$options$ruleTyp, _this$options$ruleTyp2;
    this.maxAlerts = maxAlerts;
    this.flappingSettings = flappingSettings;
    this.ruleLogPrefix = ruleLabel;
    this.startedAtString = startedAt ? startedAt.toISOString() : null;
    for (const id of (0, _lodash.keys)(activeAlertsFromState)) {
      this.trackedAlerts.active[id] = new _alert.Alert(id, activeAlertsFromState[id]);
    }
    for (const id of (0, _lodash.keys)(recoveredAlertsFromState)) {
      this.trackedAlerts.recovered[id] = new _alert.Alert(id, recoveredAlertsFromState[id]);
    }

    // Legacy alerts client creates a copy of the active tracked alerts
    // This copy is updated when rule executors report alerts back to the framework
    // while the original alert is preserved
    this.reportedAlerts = (0, _lodash.cloneDeep)(this.trackedAlerts.active);
    this.alertFactory = (0, _create_alert_factory.createAlertFactory)({
      alerts: this.reportedAlerts,
      logger: this.options.logger,
      maxAlerts: this.maxAlerts,
      autoRecoverAlerts: (_this$options$ruleTyp = this.options.ruleType.autoRecoverAlerts) !== null && _this$options$ruleTyp !== void 0 ? _this$options$ruleTyp : true,
      canSetRecoveryContext: (_this$options$ruleTyp2 = this.options.ruleType.doesSetRecoveryContext) !== null && _this$options$ruleTyp2 !== void 0 ? _this$options$ruleTyp2 : false
    });
  }
  getTrackedAlerts() {
    return this.trackedAlerts;
  }
  getAlert(id) {
    var _this$alertFactory;
    return (_this$alertFactory = this.alertFactory) === null || _this$alertFactory === void 0 ? void 0 : _this$alertFactory.get(id);
  }
  isTrackedAlert(id) {
    return !!this.trackedAlerts.active[id];
  }
  async processAlerts({
    flappingSettings,
    alertDelay,
    ruleRunMetricsStore
  }) {
    var _this$options$ruleTyp3;
    const {
      newAlerts: processedAlertsNew,
      activeAlerts: processedAlertsActive,
      currentRecoveredAlerts: processedAlertsRecoveredCurrent,
      recoveredAlerts: processedAlertsRecovered
    } = (0, _lib.processAlerts)({
      alerts: this.reportedAlerts,
      existingAlerts: this.trackedAlerts.active,
      previouslyRecoveredAlerts: this.trackedAlerts.recovered,
      hasReachedAlertLimit: this.alertFactory.hasReachedAlertLimit(),
      alertLimit: this.maxAlerts,
      autoRecoverAlerts: (_this$options$ruleTyp3 = this.options.ruleType.autoRecoverAlerts) !== null && _this$options$ruleTyp3 !== void 0 ? _this$options$ruleTyp3 : true,
      flappingSettings,
      startedAt: this.startedAtString
    });
    if (this.options.maintenanceWindowsService) {
      // load maintenance windows if there are any any alerts (new, active, recovered)
      // this is because we need the MW IDs for any active or recovered alerts that may
      // have started during the MW period.
      if ((0, _lodash.keys)(processedAlertsNew).length > 0 || (0, _lodash.keys)(processedAlertsActive).length > 0 || (0, _lodash.keys)(processedAlertsRecovered).length > 0) {
        const {
          maintenanceWindowsWithoutScopedQueryIds,
          maintenanceWindows
        } = await this.options.maintenanceWindowsService.getMaintenanceWindows({
          eventLogger: this.options.alertingEventLogger,
          request: this.options.request,
          ruleTypeCategory: this.options.ruleType.category,
          spaceId: this.options.spaceId
        });
        this.removeExpiredMaintenanceWindows({
          processedAlertsActive,
          processedAlertsRecovered,
          maintenanceWindows
        });
        for (const id in processedAlertsNew) {
          if (Object.hasOwn(processedAlertsNew, id)) {
            processedAlertsNew[id].setMaintenanceWindowIds(maintenanceWindowsWithoutScopedQueryIds);
          }
        }
      }
    }
    const {
      trimmedAlertsRecovered,
      earlyRecoveredAlerts
    } = (0, _trim_recovered_alerts.trimRecoveredAlerts)(this.options.logger, processedAlertsRecovered, this.maxAlerts);
    const alerts = (0, _lib.getAlertsForNotification)(flappingSettings, this.options.ruleType.defaultActionGroupId, alertDelay, processedAlertsNew, processedAlertsActive, trimmedAlertsRecovered, processedAlertsRecoveredCurrent, this.startedAtString);
    ruleRunMetricsStore.setNumberOfDelayedAlerts(alerts.delayedAlertsCount);
    alerts.currentRecoveredAlerts = (0, _lodash.merge)(alerts.currentRecoveredAlerts, earlyRecoveredAlerts);
    this.processedAlerts.new = alerts.newAlerts;
    this.processedAlerts.active = alerts.activeAlerts;
    this.processedAlerts.activeCurrent = alerts.currentActiveAlerts;
    this.processedAlerts.recovered = alerts.recoveredAlerts;
    this.processedAlerts.recoveredCurrent = alerts.currentRecoveredAlerts;
  }
  logAlerts({
    ruleRunMetricsStore,
    shouldLogAlerts
  }) {
    var _this$options$ruleTyp4;
    (0, _log_alerts.logAlerts)({
      logger: this.options.logger,
      alertingEventLogger: this.options.alertingEventLogger,
      newAlerts: this.processedAlerts.new,
      activeAlerts: this.processedAlerts.activeCurrent,
      recoveredAlerts: this.processedAlerts.recoveredCurrent,
      ruleLogPrefix: this.ruleLogPrefix,
      ruleRunMetricsStore,
      canSetRecoveryContext: (_this$options$ruleTyp4 = this.options.ruleType.doesSetRecoveryContext) !== null && _this$options$ruleTyp4 !== void 0 ? _this$options$ruleTyp4 : false,
      shouldPersistAlerts: shouldLogAlerts
    });
  }
  getProcessedAlerts(type) {
    if (Object.hasOwn(this.processedAlerts, type)) {
      return this.processedAlerts[type];
    }
    return {};
  }
  getAlertsToSerialize(shouldSetFlappingAndOptimize = true) {
    if (shouldSetFlappingAndOptimize) {
      (0, _lib.setFlapping)(this.flappingSettings, this.processedAlerts.active, this.processedAlerts.recovered);
    }
    return (0, _lib.determineAlertsToReturn)(this.processedAlerts.active, this.processedAlerts.recovered, shouldSetFlappingAndOptimize);
  }
  hasReachedAlertLimit() {
    return this.alertFactory.hasReachedAlertLimit();
  }
  getMaxAlertLimit() {
    return this.maxAlerts;
  }
  checkLimitUsage() {
    return this.alertFactory.alertLimit.checkLimitUsage();
  }
  factory() {
    return (0, _create_alert_factory.getPublicAlertFactory)(this.alertFactory);
  }
  client() {
    return null;
  }
  async persistAlerts() {
    return;
  }
  async updatePersistedAlertsWithMaintenanceWindowIds() {
    return null;
  }
  async setAlertStatusToUntracked() {
    return;
  }
  getTrackedExecutions() {
    return new Set([]);
  }
  removeExpiredMaintenanceWindows({
    processedAlertsActive,
    processedAlertsRecovered,
    maintenanceWindows
  }) {
    const maintenanceWindowIds = maintenanceWindows.map(mw => mw.id);
    const clearMws = alerts => {
      for (const id in alerts) {
        if (Object.hasOwn(alerts, id)) {
          const existingMaintenanceWindowIds = alerts[id].getMaintenanceWindowIds();
          const activeMaintenanceWindowIds = existingMaintenanceWindowIds.filter(mw => {
            return maintenanceWindowIds.includes(mw);
          });
          alerts[id].setMaintenanceWindowIds(activeMaintenanceWindowIds);
        }
      }
    };
    clearMws(processedAlertsActive);
    clearMws(processedAlertsRecovered);
  }
}
exports.LegacyAlertsClient = LegacyAlertsClient;