"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getHoverItem = getHoverItem;
var _i18n = require("@kbn/i18n");
var _esqlValidationAutocomplete = require("@kbn/esql-validation-autocomplete");
var _helpers = require("@kbn/esql-validation-autocomplete/src/shared/helpers");
var _helper = require("@kbn/esql-validation-autocomplete/src/autocomplete/helper");
var _resources_helpers = require("@kbn/esql-validation-autocomplete/src/shared/resources_helpers");
var _autocomplete = require("@kbn/esql-validation-autocomplete/src/autocomplete/autocomplete");
var _factories = require("@kbn/esql-validation-autocomplete/src/autocomplete/factories");
var _types = require("@kbn/esql-ast/src/types");
var _utils = require("../shared/utils");
var _monaco_imports = require("../../../monaco_imports");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const ACCEPTABLE_TYPES_HOVER = _i18n.i18n.translate('monaco.esql.hover.acceptableTypes', {
  defaultMessage: 'Acceptable types'
});
async function getHoverItemForFunction(model, position, token, astProvider, resourceRetriever) {
  const context = {
    triggerCharacter: ' ',
    triggerKind: 1
  };
  const fullText = model.getValue();
  const offset = (0, _utils.monacoPositionToOffset)(fullText, position);
  const innerText = fullText.substring(0, offset);
  const correctedQuery = (0, _helpers.correctQuerySyntax)(innerText, context);
  const {
    ast
  } = await astProvider(correctedQuery);
  const astContext = (0, _esqlValidationAutocomplete.getAstContext)(innerText, ast, offset);
  const {
    node
  } = astContext;
  const commands = ast;
  if ((0, _types.isESQLFunction)(node) && astContext.type === 'function') {
    const queryForFields = (0, _helper.getQueryForFields)((0, _resources_helpers.buildQueryUntilPreviousCommand)(ast, correctedQuery), ast);
    const {
      getFieldsMap
    } = (0, _autocomplete.getFieldsByTypeRetriever)(queryForFields, resourceRetriever);
    const fnDefinition = (0, _esqlValidationAutocomplete.getFunctionDefinition)(node.name);
    // early exit on no hit
    if (!fnDefinition) {
      return undefined;
    }
    const fieldsMap = await getFieldsMap();
    const anyVariables = (0, _esqlValidationAutocomplete.collectVariables)(commands, fieldsMap, innerText);
    const references = {
      fields: fieldsMap,
      variables: anyVariables
    };
    const {
      typesToSuggestNext,
      enrichedArgs
    } = (0, _helper.getValidSignaturesAndTypesToSuggestNext)(node, references, fnDefinition, fullText, offset);
    const hoveredArg = enrichedArgs[enrichedArgs.length - 1];
    const contents = [];
    if (hoveredArg && (0, _types.isESQLNamedParamLiteral)(hoveredArg)) {
      const bestMatch = _factories.TIME_SYSTEM_PARAMS.find(p => p.startsWith(hoveredArg.text));
      // We only know if it's start or end after first 3 characters (?t_s or ?t_e)
      if (hoveredArg.text.length > 3 && bestMatch) {
        Object.entries(_factories.TIME_SYSTEM_DESCRIPTIONS).forEach(([key, value]) => {
          contents.push({
            value: `**${key}**: ${value}`
          });
        });
      }
    }
    if (typesToSuggestNext.length > 0) {
      contents.push({
        value: `**${ACCEPTABLE_TYPES_HOVER}**: ${typesToSuggestNext.map(({
          type,
          constantOnly
        }) => `${constantOnly ? '_constant_ ' : ''}**${type}**` + (
        // If function arg is a constant date, helpfully suggest named time system params
        constantOnly && type === 'date' ? ` | ${_factories.TIME_SYSTEM_PARAMS.join(' | ')}` : '')).join(' | ')}`
      });
    }
    const hints = contents.length > 0 ? {
      range: new _monaco_imports.monaco.Range(1, 1, model.getLineCount(), model.getLineMaxColumn(model.getLineCount())),
      contents
    } : undefined;
    return hints;
  }
}
async function getHoverItem(model, position, token, astProvider, resourceRetriever) {
  const fullText = model.getValue();
  const offset = (0, _utils.monacoPositionToOffset)(fullText, position);
  const {
    ast
  } = await astProvider(fullText);
  const astContext = (0, _esqlValidationAutocomplete.getAstContext)(fullText, ast, offset);
  const {
    getPolicyMetadata
  } = (0, _esqlValidationAutocomplete.getPolicyHelper)(resourceRetriever);
  let hoverContent = {
    contents: []
  };
  const hoverItemsForFunction = await getHoverItemForFunction(model, position, token, astProvider, resourceRetriever);
  if (hoverItemsForFunction) {
    hoverContent = hoverItemsForFunction;
  }
  if (['newCommand', 'list'].includes(astContext.type)) {
    return {
      contents: []
    };
  }
  if (astContext.type === 'function') {
    const fnDefinition = (0, _esqlValidationAutocomplete.getFunctionDefinition)(astContext.node.name);
    if (fnDefinition) {
      hoverContent.contents.push(...[{
        value: (0, _esqlValidationAutocomplete.getFunctionSignatures)(fnDefinition)[0].declaration
      }, {
        value: fnDefinition.description
      }]);
    }
  }
  if (astContext.type === 'expression') {
    if (astContext.node) {
      if ((0, _esqlValidationAutocomplete.isSourceItem)(astContext.node) && astContext.node.sourceType === 'policy') {
        const policyMetadata = await getPolicyMetadata(astContext.node.name);
        if (policyMetadata) {
          hoverContent.contents.push(...[{
            value: `${_i18n.i18n.translate('monaco.esql.hover.policyIndexes', {
              defaultMessage: '**Indexes**'
            })}: ${policyMetadata.sourceIndices.join(', ')}`
          }, {
            value: `${_i18n.i18n.translate('monaco.esql.hover.policyMatchingField', {
              defaultMessage: '**Matching field**'
            })}: ${policyMetadata.matchField}`
          }, {
            value: `${_i18n.i18n.translate('monaco.esql.hover.policyEnrichedFields', {
              defaultMessage: '**Fields**'
            })}: ${policyMetadata.enrichFields.join(', ')}`
          }]);
        }
      }
      if ((0, _esqlValidationAutocomplete.isSettingItem)(astContext.node)) {
        const commandDef = (0, _esqlValidationAutocomplete.getCommandDefinition)(astContext.command.name);
        const settingDef = commandDef === null || commandDef === void 0 ? void 0 : commandDef.modes.find(({
          values
        }) => values.some(({
          name
        }) => name === astContext.node.name));
        if (settingDef) {
          const mode = settingDef.values.find(({
            name
          }) => name === astContext.node.name);
          hoverContent.contents.push(...[{
            value: settingDef.description
          }, {
            value: `**${mode.name}**: ${mode.description}`
          }]);
        }
      }
    }
  }
  return hoverContent;
}