"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ESQLLang = void 0;
var _monaco_imports = require("../monaco_imports");
var _constants = require("./lib/constants");
var _worker_proxy = require("../common/worker_proxy");
var _esql_ast_provider = require("./lib/esql_ast_provider");
var _suggestions = require("./lib/converters/suggestions");
var _actions = require("./lib/converters/actions");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
const workerProxyService = new _worker_proxy.WorkerProxyService();
const removeKeywordSuffix = name => {
  return name.endsWith('.keyword') ? name.slice(0, -8) : name;
};
const ESQLLang = exports.ESQLLang = {
  ID: _constants.ESQL_LANG_ID,
  async onLanguage() {
    const {
      ESQLTokensProvider
    } = await Promise.resolve().then(() => _interopRequireWildcard(require('./lib')));
    workerProxyService.setup(_constants.ESQL_LANG_ID);
    _monaco_imports.monaco.languages.setTokensProvider(_constants.ESQL_LANG_ID, new ESQLTokensProvider());
  },
  languageConfiguration: {
    brackets: [['(', ')'], ['[', ']']],
    autoClosingPairs: [{
      open: '(',
      close: ')'
    }, {
      open: '[',
      close: ']'
    }, {
      open: `'`,
      close: `'`
    }, {
      open: '"""',
      close: '"""'
    }, {
      open: '"',
      close: '"'
    }],
    surroundingPairs: [{
      open: '(',
      close: ')'
    }, {
      open: `'`,
      close: `'`
    }, {
      open: '"""',
      close: '"""'
    }, {
      open: '"',
      close: '"'
    }]
  },
  validate: async (model, code, callbacks) => {
    const astAdapter = new _esql_ast_provider.ESQLAstAdapter((...uris) => workerProxyService.getWorker(uris), callbacks);
    return await astAdapter.validate(model, code);
  },
  getSignatureProvider: callbacks => {
    return {
      signatureHelpTriggerCharacters: [' ', '('],
      async provideSignatureHelp(model, position, _token, context) {
        const astAdapter = new _esql_ast_provider.ESQLAstAdapter((...uris) => workerProxyService.getWorker(uris), callbacks);
        return astAdapter.suggestSignature(model, position, context);
      }
    };
  },
  getHoverProvider: callbacks => {
    return {
      async provideHover(model, position, token) {
        const astAdapter = new _esql_ast_provider.ESQLAstAdapter((...uris) => workerProxyService.getWorker(uris), callbacks);
        return astAdapter.getHover(model, position, token);
      }
    };
  },
  getSuggestionProvider: callbacks => {
    return {
      triggerCharacters: [',', '(', '=', ' ', '[', ''],
      async provideCompletionItems(model, position, context) {
        const astAdapter = new _esql_ast_provider.ESQLAstAdapter((...uris) => workerProxyService.getWorker(uris), callbacks);
        const suggestions = await astAdapter.autocomplete(model, position, context);
        return {
          // @ts-expect-error because of range typing: https://github.com/microsoft/monaco-editor/issues/4638
          suggestions: (0, _suggestions.wrapAsMonacoSuggestions)(suggestions)
        };
      },
      async resolveCompletionItem(item, token) {
        if (!(callbacks !== null && callbacks !== void 0 && callbacks.getFieldsMetadata)) return item;
        const fieldsMetadataClient = await (callbacks === null || callbacks === void 0 ? void 0 : callbacks.getFieldsMetadata);
        const fullEcsMetadataList = await (fieldsMetadataClient === null || fieldsMetadataClient === void 0 ? void 0 : fieldsMetadataClient.find({
          attributes: ['type']
        }));
        if (!fullEcsMetadataList || !fieldsMetadataClient || typeof item.label !== 'string') return item;
        const strippedFieldName = removeKeywordSuffix(item.label);
        if (
        // If item is not a field, no need to fetch metadata
        item.kind === _monaco_imports.monaco.languages.CompletionItemKind.Variable &&
        // If not ECS, no need to fetch description
        Object.hasOwn(fullEcsMetadataList === null || fullEcsMetadataList === void 0 ? void 0 : fullEcsMetadataList.fields, strippedFieldName)) {
          const ecsMetadata = await fieldsMetadataClient.find({
            fieldNames: [strippedFieldName],
            attributes: ['description']
          });
          const fieldMetadata = ecsMetadata.fields[strippedFieldName];
          if (fieldMetadata && fieldMetadata.description) {
            const completionItem = {
              ...item,
              documentation: {
                value: fieldMetadata.description
              }
            };
            return completionItem;
          }
        }
        return item;
      }
    };
  },
  getCodeActionProvider: callbacks => {
    return {
      async provideCodeActions(model /** ITextModel*/, range /** Range*/, context /** CodeActionContext*/, token /** CancellationToken*/) {
        const astAdapter = new _esql_ast_provider.ESQLAstAdapter((...uris) => workerProxyService.getWorker(uris), callbacks);
        const actions = await astAdapter.codeAction(model, range, context);
        return {
          actions: (0, _actions.wrapAsMonacoCodeActions)(model, actions),
          dispose: () => {}
        };
      }
    };
  }
};