"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.filterForEnabledFeatureModels = filterForEnabledFeatureModels;
exports.populateInferenceServicesProvider = void 0;
exports.trainedModelsRoutes = trainedModelsRoutes;
var _lodash = require("lodash");
var _configSchema = require("@kbn/config-schema");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _trained_models = require("../../common/constants/trained_models");
var _app = require("../../common/constants/app");
var _error_wrapper = require("../client/error_wrapper");
var _inference_schema = require("./schemas/inference_schema");
var _log = require("../lib/log");
var _anomaly_detectors_schema = require("./schemas/anomaly_detectors_schema");
var _model_management = require("../models/model_management");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function filterForEnabledFeatureModels(models, enabledFeatures) {
  let filteredModels = models;
  if (enabledFeatures.nlp === false) {
    filteredModels = filteredModels.filter(m => m.model_type === 'tree_ensemble');
  }
  if (enabledFeatures.dfa === false) {
    filteredModels = filteredModels.filter(m => m.model_type !== 'tree_ensemble');
  }
  return filteredModels;
}
const populateInferenceServicesProvider = client => {
  return async function populateInferenceServices(trainedModels, asInternal = false) {
    const esClient = asInternal ? client.asInternalUser : client.asCurrentUser;
    try {
      // Check if model is used by an inference service
      const {
        endpoints
      } = await esClient.transport.request({
        method: 'GET',
        path: `/_inference/_all`
      });
      const inferenceAPIMap = (0, _lodash.groupBy)(endpoints, endpoint => endpoint.service === 'elser' && endpoint.service_settings.model_id);
      for (const model of trainedModels) {
        const inferenceApis = inferenceAPIMap[model.model_id];
        model.hasInferenceServices = !!inferenceApis;
        if (model.hasInferenceServices && !asInternal) {
          model.inference_apis = inferenceApis;
        }
      }
    } catch (e) {
      if (!asInternal && e.statusCode === 403) {
        // retry with internal user to get an indicator if models has associated inference services, without mentioning the names
        await populateInferenceServices(trainedModels, true);
      } else {
        _log.mlLog.error(e);
      }
    }
  };
};
exports.populateInferenceServicesProvider = populateInferenceServicesProvider;
function trainedModelsRoutes({
  router,
  routeGuard,
  getEnabledFeatures
}, cloud) {
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/{modelId?}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetTrainedModels']
    },
    summary: 'Get info of a trained inference model',
    description: 'Retrieves configuration information for a trained model.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.optionalModelIdSchema,
        query: _inference_schema.getInferenceQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        modelId
      } = request.params;
      const {
        with_pipelines: withPipelines,
        with_indices: withIndicesRaw,
        ...getTrainedModelsRequestParams
      } = request.query;
      const withIndices = request.query.with_indices === 'true' || request.query.with_indices === true;
      const resp = await mlClient.getTrainedModels({
        ...getTrainedModelsRequestParams,
        ...(modelId ? {
          model_id: modelId
        } : {}),
        size: _trained_models.DEFAULT_TRAINED_MODELS_PAGE_SIZE
      });
      // model_type is missing
      // @ts-ignore
      const result = resp.trained_model_configs;
      const populateInferenceServices = populateInferenceServicesProvider(client);
      await populateInferenceServices(result, false);
      try {
        if (withPipelines) {
          // Also need to retrieve the list of deployment IDs from stats
          const stats = await mlClient.getTrainedModelsStats({
            ...(modelId ? {
              model_id: modelId
            } : {}),
            size: 10000
          });
          const modelDeploymentsMap = stats.trained_model_stats.reduce((acc, curr) => {
            if (!curr.deployment_stats) return acc;
            // @ts-ignore elasticsearch-js client is missing deployment_id
            const deploymentId = curr.deployment_stats.deployment_id;
            if (acc[curr.model_id]) {
              acc[curr.model_id].push(deploymentId);
            } else {
              acc[curr.model_id] = [deploymentId];
            }
            return acc;
          }, {});
          const modelIdsAndAliases = Array.from(new Set([...result.map(({
            model_id: id,
            metadata
          }) => {
            var _metadata$model_alias;
            return [id, ...((_metadata$model_alias = metadata === null || metadata === void 0 ? void 0 : metadata.model_aliases) !== null && _metadata$model_alias !== void 0 ? _metadata$model_alias : [])];
          }).flat(), ...Object.values(modelDeploymentsMap).flat()]));
          const modelsClient = (0, _model_management.modelsProvider)(client, mlClient, cloud);
          const modelsPipelinesAndIndices = await Promise.all(modelIdsAndAliases.map(async modelIdOrAlias => {
            return {
              modelIdOrAlias,
              result: await modelsClient.getModelsPipelinesAndIndicesMap(modelIdOrAlias, {
                withIndices
              })
            };
          }));
          for (const model of result) {
            var _model$metadata$model, _model$metadata, _modelsPipelinesAndIn;
            const modelAliases = (_model$metadata$model = (_model$metadata = model.metadata) === null || _model$metadata === void 0 ? void 0 : _model$metadata.model_aliases) !== null && _model$metadata$model !== void 0 ? _model$metadata$model : [];
            const modelMap = (_modelsPipelinesAndIn = modelsPipelinesAndIndices.find(d => d.modelIdOrAlias === model.model_id)) === null || _modelsPipelinesAndIn === void 0 ? void 0 : _modelsPipelinesAndIn.result;
            const allRelatedModels = modelsPipelinesAndIndices.filter(m => {
              var _modelDeploymentsMap$;
              return [model.model_id, ...modelAliases, ...((_modelDeploymentsMap$ = modelDeploymentsMap[model.model_id]) !== null && _modelDeploymentsMap$ !== void 0 ? _modelDeploymentsMap$ : [])].findIndex(alias => alias === m.modelIdOrAlias) > -1;
            }).map(r => r === null || r === void 0 ? void 0 : r.result).filter(_mlIsDefined.isDefined);
            const ingestPipelinesFromModelAliases = allRelatedModels.map(r => r === null || r === void 0 ? void 0 : r.ingestPipelines).filter(_mlIsDefined.isDefined);
            model.pipelines = ingestPipelinesFromModelAliases.reduce((allPipelines, modelsToPipelines) => {
              for (const [, pipelinesObj] of modelsToPipelines === null || modelsToPipelines === void 0 ? void 0 : modelsToPipelines.entries()) {
                Object.entries(pipelinesObj).forEach(([pipelineId, pipelineInfo]) => {
                  allPipelines[pipelineId] = pipelineInfo;
                });
              }
              return allPipelines;
            }, {});
            if (modelMap && withIndices) {
              model.indices = modelMap.indices;
            }
          }
        }
      } catch (e) {
        // the user might not have required permissions to fetch pipelines
        // log the error to the debug log as this might be a common situation and
        // we don't need to fill kibana's log with these messages.
        _log.mlLog.debug(e);
      }
      const filteredModels = filterForEnabledFeatureModels(result, getEnabledFeatures());
      try {
        const jobIds = filteredModels.map(model => {
          var _model$metadata2, _model$metadata2$anal;
          const id = (_model$metadata2 = model.metadata) === null || _model$metadata2 === void 0 ? void 0 : (_model$metadata2$anal = _model$metadata2.analytics_config) === null || _model$metadata2$anal === void 0 ? void 0 : _model$metadata2$anal.id;
          if (id) {
            return `${id}*`;
          }
        }).filter(id => id !== undefined);
        if (jobIds.length) {
          const {
            data_frame_analytics: jobs
          } = await mlClient.getDataFrameAnalytics({
            id: jobIds.join(','),
            allow_no_match: true
          });
          filteredModels.forEach(model => {
            var _model$metadata3, _model$metadata3$anal;
            const dfaId = model === null || model === void 0 ? void 0 : (_model$metadata3 = model.metadata) === null || _model$metadata3 === void 0 ? void 0 : (_model$metadata3$anal = _model$metadata3.analytics_config) === null || _model$metadata3$anal === void 0 ? void 0 : _model$metadata3$anal.id;
            if (dfaId !== undefined) {
              // if this is a dfa model, set origin_job_exists
              model.origin_job_exists = jobs.find(job => job.id === dfaId) !== undefined;
            }
          });
        }
      } catch (e) {
        // Swallow error to prevent blocking trained models result
      }
      return response.ok({
        body: filteredModels
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/_stats`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetTrainedModels']
    },
    summary: 'Get stats for all trained models',
    description: 'Retrieves usage information for all trained models.'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const body = await mlClient.getTrainedModelsStats({
        size: _trained_models.DEFAULT_TRAINED_MODELS_PAGE_SIZE
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/{modelId}/_stats`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetTrainedModels']
    },
    summary: 'Get stats for a trained model',
    description: 'Retrieves usage information for a trained model.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.modelIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        modelId
      } = request.params;
      const body = await mlClient.getTrainedModelsStats({
        ...(modelId ? {
          model_id: modelId
        } : {})
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/{modelId}/pipelines`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetTrainedModels']
    },
    summary: 'Get trained model pipelines',
    description: 'Retrieves ingest pipelines associated with a trained model.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.modelIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    mlClient,
    response
  }) => {
    try {
      const {
        modelId
      } = request.params;
      const result = await (0, _model_management.modelsProvider)(client, mlClient, cloud).getModelsPipelines(modelId.split(','));
      return response.ok({
        body: [...result].map(([id, pipelines]) => ({
          model_id: id,
          pipelines
        }))
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/ingest_pipelines`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetTrainedModels'] // TODO: update permissions
    },
    summary: 'Get ingest pipelines',
    description: 'Retrieves ingest pipelines.'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    mlClient,
    response
  }) => {
    try {
      const body = await (0, _model_management.modelsProvider)(client, mlClient, cloud).getPipelines();
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/create_inference_pipeline`,
    access: 'internal',
    options: {
      tags: ['access:ml:canCreateTrainedModels']
    },
    summary: 'Create an inference pipeline',
    description: 'Creates a pipeline with inference processor'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _inference_schema.createIngestPipelineSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    mlClient,
    response
  }) => {
    try {
      const {
        pipeline,
        pipelineName
      } = request.body;
      const body = await (0, _model_management.modelsProvider)(client, mlClient, cloud).createInferencePipeline(pipeline, pipelineName);
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.put({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/{modelId}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canCreateTrainedModels']
    },
    summary: 'Put a trained model',
    description: 'Adds a new trained model'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.modelIdSchema,
        body: _configSchema.schema.any(),
        query: _inference_schema.putTrainedModelQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      var _request$query;
      const {
        modelId
      } = request.params;
      const body = await mlClient.putTrainedModel({
        model_id: modelId,
        body: request.body,
        ...((_request$query = request.query) !== null && _request$query !== void 0 && _request$query.defer_definition_decompression ? {
          defer_definition_decompression: true
        } : {})
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.delete({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/{modelId}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canDeleteTrainedModels']
    },
    summary: 'Delete a trained model',
    description: 'Deletes an existing trained model that is currently not referenced by an ingest pipeline.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.modelIdSchema,
        query: _inference_schema.deleteTrainedModelQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response,
    client
  }) => {
    try {
      const {
        modelId
      } = request.params;
      const {
        with_pipelines: withPipelines,
        force
      } = request.query;
      if (withPipelines) {
        // first we need to delete pipelines, otherwise ml api return an error
        await (0, _model_management.modelsProvider)(client, mlClient, cloud).deleteModelPipelines(modelId.split(','));
      }
      const body = await mlClient.deleteTrainedModel({
        model_id: modelId,
        force
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/{modelId}/deployment/_start`,
    access: 'internal',
    options: {
      tags: ['access:ml:canStartStopTrainedModels']
    },
    summary: 'Start trained model deployment',
    description: 'Starts trained model deployment.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.modelIdSchema,
        query: _inference_schema.threadingParamsQuerySchema,
        body: _inference_schema.threadingParamsBodySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        modelId
      } = request.params;

      // TODO use mlClient.startTrainedModelDeployment when esClient is updated
      const body = await mlClient.startTrainedModelDeployment({
        model_id: modelId,
        ...(request.query ? request.query : {}),
        ...(request.body ? request.body : {})
      }, {
        maxRetries: 0
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/{modelId}/{deploymentId}/deployment/_update`,
    access: 'internal',
    options: {
      tags: ['access:ml:canStartStopTrainedModels']
    },
    summary: 'Update trained model deployment',
    description: 'Updates trained model deployment.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.modelAndDeploymentIdSchema,
        body: _inference_schema.updateDeploymentParamsSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        modelId,
        deploymentId
      } = request.params;
      const body = await mlClient.updateTrainedModelDeployment({
        model_id: modelId,
        deployment_id: deploymentId,
        ...request.body
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/{modelId}/{deploymentId}/deployment/_stop`,
    access: 'internal',
    options: {
      tags: ['access:ml:canStartStopTrainedModels']
    },
    summary: 'Stop trained model deployment',
    description: 'Stops trained model deployment.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.modelAndDeploymentIdSchema,
        query: _anomaly_detectors_schema.forceQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        deploymentId,
        modelId
      } = request.params;
      const results = Object.create(null);
      for (const id of deploymentId.split(',')) {
        try {
          var _request$query$force;
          const {
            stopped: success
          } = await mlClient.stopTrainedModelDeployment({
            model_id: modelId,
            deployment_id: id,
            force: (_request$query$force = request.query.force) !== null && _request$query$force !== void 0 ? _request$query$force : false,
            allow_no_match: false
          });
          results[id] = {
            success
          };
        } catch (error) {
          results[id] = {
            success: false,
            error
          };
        }
      }
      return response.ok({
        body: results
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/pipeline_simulate`,
    access: 'internal',
    options: {
      tags: ['access:ml:canTestTrainedModels']
    },
    summary: 'Simulates an ingest pipeline',
    description: 'Simulates an ingest pipeline.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _inference_schema.pipelineSimulateBody
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    response
  }) => {
    try {
      const {
        pipeline,
        docs
      } = request.body;
      const body = await client.asInternalUser.ingest.simulate({
        pipeline,
        docs
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/infer/{modelId}/{deploymentId}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canTestTrainedModels']
    },
    summary: 'Evaluates a trained model.',
    description: 'Evaluates a trained model.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.modelAndDeploymentIdSchema,
        query: _inference_schema.inferTrainedModelQuery,
        body: _inference_schema.inferTrainedModelBody
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    request,
    response
  }) => {
    try {
      const {
        modelId,
        deploymentId
      } = request.params;
      const body = await mlClient.inferTrainedModel({
        model_id: modelId,
        deployment_id: deploymentId,
        body: {
          docs: request.body.docs,
          ...(request.body.inference_config ? {
            inference_config: request.body.inference_config
          } : {})
        },
        ...(request.query.timeout ? {
          timeout: request.query.timeout
        } : {})
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/model_downloads`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetTrainedModels']
    },
    summary: 'Get available models for download',
    description: 'Gets available models for download with supported and recommended flags based on the cluster OS and CPU architecture.'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.fullLicenseAPIGuard(async ({
    response,
    mlClient,
    client
  }) => {
    try {
      const body = await (0, _model_management.modelsProvider)(client, mlClient, cloud).getModelDownloads();
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/elser_config`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetTrainedModels']
    },
    summary: 'Get ELSER config for download',
    description: 'Gets ELSER config for download based on the cluster OS and CPU architecture.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        query: _inference_schema.modelDownloadsQuery
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    response,
    client,
    mlClient,
    request
  }) => {
    try {
      const {
        version
      } = request.query;
      const body = await (0, _model_management.modelsProvider)(client, mlClient, cloud).getELSER(version ? {
        version: Number(version)
      } : undefined);
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/install_elastic_trained_model/{modelId}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canCreateTrainedModels']
    },
    summary: 'Install Elastic trained model',
    description: 'Downloads and installs Elastic trained model.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.modelIdSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    mlSavedObjectService
  }) => {
    try {
      const {
        modelId
      } = request.params;
      const body = await (0, _model_management.modelsProvider)(client, mlClient, cloud).installElasticModel(modelId, mlSavedObjectService);
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/download_status`,
    access: 'internal',
    options: {
      tags: ['access:ml:canCreateTrainedModels']
    },
    summary: 'Get models download status',
    description: 'Gets download status for all currently downloading models.'
  }).addVersion({
    version: '1',
    validate: false
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    mlSavedObjectService
  }) => {
    try {
      const body = await (0, _model_management.modelsProvider)(client, mlClient, cloud).getModelsDownloadStatus();
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/curated_model_config/{modelName}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetTrainedModels']
    },
    summary: 'Get curated model config',
    description: 'Gets curated model config for the specified model based on cluster architecture.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _inference_schema.curatedModelsParamsSchema,
        query: _inference_schema.curatedModelsQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    mlSavedObjectService
  }) => {
    try {
      const body = await (0, _model_management.modelsProvider)(client, mlClient, cloud).getCuratedModelConfig(request.params.modelName, {
        version: request.query.version
      });
      return response.ok({
        body
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}