"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dataRecognizer = dataRecognizer;
var _app = require("../../common/constants/app");
var _error_wrapper = require("../client/error_wrapper");
var _data_recognizer = require("../models/data_recognizer");
var _modules = require("./schemas/modules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Recognizer routes.
 */
function dataRecognizer({
  router,
  routeGuard
}, compatibleModuleType) {
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/modules/recognize/{indexPatternTitle}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canCreateJob']
    },
    summary: 'Recognize index pattern',
    description: 'By supplying an index pattern, discover if any of the modules are a match for data in that index.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _modules.recognizeModulesSchema,
        query: _modules.moduleFilterSchema
      },
      response: {
        200: {
          body: _modules.recognizeModulesSchemaResponse
        }
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    context,
    mlSavedObjectService,
    getDataViewsService
  }) => {
    try {
      var _request$query$filter;
      const {
        indexPatternTitle
      } = request.params;
      const filter = (_request$query$filter = request.query.filter) === null || _request$query$filter === void 0 ? void 0 : _request$query$filter.split(',');
      const soClient = (await context.core).savedObjects.client;
      const dataViewsService = await getDataViewsService();
      const dr = (0, _data_recognizer.dataRecognizerFactory)(client, mlClient, soClient, dataViewsService, mlSavedObjectService, request, compatibleModuleType);
      const results = await dr.findMatches(indexPatternTitle, filter);
      return response.ok({
        body: results
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/modules/recognize_by_module/{moduleId}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canCreateJob']
    },
    summary: 'Recognize module',
    description: 'By supplying a module id, discover if any of the data views contain data that is a match for that module.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _modules.moduleIdParamSchema,
        query: _modules.optionalSizeQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    context,
    mlSavedObjectService,
    getDataViewsService
  }) => {
    try {
      const {
        moduleId
      } = request.params;
      const {
        size
      } = request.query;
      const soClient = (await context.core).savedObjects.client;
      const dataViewsService = await getDataViewsService();
      const dr = (0, _data_recognizer.dataRecognizerFactory)(client, mlClient, soClient, dataViewsService, mlSavedObjectService, request, compatibleModuleType);
      const results = await dr.findIndexMatches(moduleId, size);
      return response.ok({
        body: results
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/modules/get_module/{moduleId?}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetJobs']
    },
    summary: 'Get module',
    description: 'Retrieve a whole ML module, containing jobs, datafeeds and saved objects. If no module ID is supplied, returns all modules.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _modules.optionalModuleIdParamSchema,
        query: _modules.moduleFilterSchema
      },
      response: {
        200: {
          body: _modules.getModulesSchemaResponse,
          description: 'When a module ID is specified, returns a module object containing all of the jobs, datafeeds and saved objects which will be created when the module is setup. If no module ID is supplied, an array of all modules will be returned.'
        }
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    context,
    mlSavedObjectService,
    getDataViewsService
  }) => {
    try {
      var _request$query$filter2;
      let {
        moduleId
      } = request.params;
      const filter = (_request$query$filter2 = request.query.filter) === null || _request$query$filter2 === void 0 ? void 0 : _request$query$filter2.split(',');
      if (moduleId === '') {
        // if the endpoint is called with a trailing /
        // the moduleId will be an empty string.
        moduleId = undefined;
      }
      const soClient = (await context.core).savedObjects.client;
      const dataViewsService = await getDataViewsService();
      const dr = (0, _data_recognizer.dataRecognizerFactory)(client, mlClient, soClient, dataViewsService, mlSavedObjectService, request, compatibleModuleType);
      const results = moduleId === undefined ? await dr.listModules(filter) : await dr.getModule(moduleId, filter);
      return response.ok({
        body: results
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/modules/setup/{moduleId}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canCreateJob']
    },
    summary: 'Setup module',
    description: 'Runs the module setup process. This creates jobs, datafeeds and kibana saved objects. It allows for customization of the module, overriding the default configuration. It also allows the user to start the datafeed.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _modules.moduleIdParamSchema,
        body: _modules.setupModuleBodySchema
      },
      response: {
        200: {
          body: _modules.dataRecognizerConfigResponse,
          description: 'An object containing the results of creating the items in a module, i.e. the jobs, datafeeds and saved objects. Each item is listed by id with a success flag signifying whether the creation was successful. If the item creation failed, an error object will also be supplied containing the error.'
        }
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    context,
    mlSavedObjectService,
    getDataViewsService
  }) => {
    try {
      const {
        moduleId
      } = request.params;
      const {
        prefix,
        groups,
        indexPatternName,
        query,
        useDedicatedIndex,
        startDatafeed,
        start,
        end,
        jobOverrides,
        datafeedOverrides,
        estimateModelMemory,
        applyToAllSpaces
      } = request.body;
      const soClient = (await context.core).savedObjects.client;
      const dataViewsService = await getDataViewsService();
      const dr = (0, _data_recognizer.dataRecognizerFactory)(client, mlClient, soClient, dataViewsService, mlSavedObjectService, request, compatibleModuleType);
      const result = await dr.setup(moduleId, prefix, groups, indexPatternName, query, useDedicatedIndex, startDatafeed, start, end, jobOverrides, datafeedOverrides, estimateModelMemory, applyToAllSpaces);
      return response.ok({
        body: result
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));

  /**
    * @apiSuccess {boolean} jobsExist <code>true</code> if all the jobs in the module have a matching job with an
   *      ID which ends with the job ID specified in the module, <code>false</code> otherwise.
   * @apiSuccess {Object[]} jobs  present if the jobs do all exist, with each object having keys of <code>id</code>,
   *      and optionally <code>earliestTimestampMs</code>, <code>latestTimestampMs</code>, <code>latestResultsTimestampMs</code>
   *      properties if the job has processed any data.
   */
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/modules/jobs_exist/{moduleId}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetJobs']
    },
    summary: 'Check if module jobs exist',
    description: `Check whether the jobs in the module with the specified ID exist in the current list of jobs. The check runs a test to see if any of the jobs in existence have an ID which ends with the ID of each job in the module. This is done as a prefix may be supplied in the setup endpoint which is added to the start of the ID of every job in the module.`
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _modules.moduleIdParamSchema
      },
      response: {
        200: {
          body: _modules.jobExistsResponse,
          description: 'jobs  present if the jobs do all exist, with each object having keys of <code>id</code>, and optionally <code>earliestTimestampMs</code>, <code>latestTimestampMs</code>, <code>latestResultsTimestampMs</code> properties if the job has processed any data.'
        }
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    context,
    mlSavedObjectService,
    getDataViewsService
  }) => {
    try {
      const {
        moduleId
      } = request.params;
      const soClient = (await context.core).savedObjects.client;
      const dataViewsService = await getDataViewsService();
      const dr = (0, _data_recognizer.dataRecognizerFactory)(client, mlClient, soClient, dataViewsService, mlSavedObjectService, request, compatibleModuleType);
      const result = await dr.dataRecognizerJobsExist(moduleId);
      return response.ok({
        body: result
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}