"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.jobAuditMessagesRoutes = jobAuditMessagesRoutes;
var _app = require("../../common/constants/app");
var _error_wrapper = require("../client/error_wrapper");
var _job_audit_messages = require("../models/job_audit_messages");
var _job_audit_messages_schema = require("./schemas/job_audit_messages_schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Routes for job audit message routes
 */
function jobAuditMessagesRoutes({
  router,
  routeGuard
}) {
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/job_audit_messages/messages/{jobId}`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetJobs']
    },
    summary: 'Gets audit messages',
    description: 'Retrieves the audit messages for the specified job ID.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _job_audit_messages_schema.jobAuditMessagesJobIdSchema,
        query: _job_audit_messages_schema.jobAuditMessagesQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    mlSavedObjectService
  }) => {
    try {
      const {
        getJobAuditMessages
      } = (0, _job_audit_messages.jobAuditMessagesProvider)(client, mlClient);
      const {
        jobId
      } = request.params;
      const {
        from,
        start,
        end
      } = request.query;
      const resp = await getJobAuditMessages(mlSavedObjectService, {
        jobId,
        from,
        start,
        end
      });
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/job_audit_messages/messages`,
    access: 'internal',
    options: {
      tags: ['access:ml:canGetJobs']
    },
    summary: 'Gets all audit messages',
    description: 'Retrieves all audit messages.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        query: _job_audit_messages_schema.jobAuditMessagesQuerySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    mlSavedObjectService
  }) => {
    try {
      const {
        getJobAuditMessages
      } = (0, _job_audit_messages.jobAuditMessagesProvider)(client, mlClient);
      const {
        from
      } = request.query;
      const resp = await getJobAuditMessages(mlSavedObjectService, {
        from
      });
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.put({
    path: `${_app.ML_INTERNAL_BASE_PATH}/job_audit_messages/clear_messages`,
    access: 'internal',
    options: {
      tags: ['access:ml:canCreateJob']
    },
    summary: 'Clear messages',
    description: 'Clear the job audit messages.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _job_audit_messages_schema.clearJobAuditMessagesBodySchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    request,
    response,
    mlSavedObjectService
  }) => {
    try {
      const {
        clearJobAuditMessages
      } = (0, _job_audit_messages.jobAuditMessagesProvider)(client, mlClient);
      const {
        jobId,
        notificationIndices
      } = request.body;
      const resp = await clearJobAuditMessages(jobId, notificationIndices);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}