"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.modelsProvider = exports.ModelsProvider = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _mlDataFrameAnalyticsUtils = require("@kbn/ml-data-frame-analytics-utils");
var _lodash = require("lodash");
var _mlTrainedModelsUtils = require("@kbn/ml-trained-models-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const modelsProvider = (client, mlClient, cloud) => new ModelsProvider(client, mlClient, cloud);
exports.modelsProvider = modelsProvider;
class ModelsProvider {
  constructor(_client, _mlClient, _cloud) {
    (0, _defineProperty2.default)(this, "_transforms", void 0);
    this._client = _client;
    this._mlClient = _mlClient;
    this._cloud = _cloud;
  }
  async initTransformData() {
    if (!this._transforms) {
      try {
        const body = await this._client.asCurrentUser.transform.getTransform({
          size: 1000
        });
        this._transforms = body.transforms;
        return body.transforms;
      } catch (e) {
        var _e$meta;
        if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) !== 403) {
          // eslint-disable-next-line no-console
          console.error(e);
        }
      }
    }
  }
  async getIndexData(index) {
    try {
      const indexData = await this._client.asInternalUser.indices.get({
        index
      });
      return indexData;
    } catch (e) {
      var _e$meta2;
      // Possible that the user doesn't have permissions to view
      // If so, gracefully exit
      if (((_e$meta2 = e.meta) === null || _e$meta2 === void 0 ? void 0 : _e$meta2.statusCode) !== 403) {
        // eslint-disable-next-line no-console
        console.error(e);
      }
      return {
        [index]: null
      };
    }
  }
  getNodeId(elementOriginalId, nodeType) {
    return `${elementOriginalId}-${nodeType}`;
  }

  /**
   * Simulates the effect of the pipeline on given document.
   *
   */
  async simulatePipeline(docs, pipelineConfig) {
    const simulateRequest = {
      docs,
      pipeline: pipelineConfig
    };
    let result = {};
    try {
      result = await this._client.asCurrentUser.ingest.simulate(simulateRequest);
    } catch (error) {
      if (error.statusCode === 404) {
        // ES returns 404 when there are no pipelines
        // Instead, we should return an empty response and a 200
        return result;
      }
      throw error;
    }
    return result;
  }

  /**
   * Creates the pipeline
   *
   */
  async createInferencePipeline(pipelineConfig, pipelineName) {
    let result = {};
    result = await this._client.asCurrentUser.ingest.putPipeline({
      id: pipelineName,
      ...pipelineConfig
    });
    return result;
  }

  /**
   * Retrieves existing pipelines.
   *
   */
  async getPipelines() {
    let result = {};
    try {
      result = await this._client.asCurrentUser.ingest.getPipeline();
    } catch (error) {
      if (error.statusCode === 404) {
        // ES returns 404 when there are no pipelines
        // Instead, we should return an empty response and a 200
        return result;
      }
      throw error;
    }
    return result;
  }

  /**
   * Retrieves the map of model ids and aliases with associated pipelines.
   * @param modelIds - Array of models ids and model aliases.
   */
  async getModelsPipelines(modelIds) {
    const modelIdsMap = new Map(modelIds.map(id => [id, null]));
    try {
      const body = await this._client.asCurrentUser.ingest.getPipeline();
      for (const [pipelineName, pipelineDefinition] of Object.entries(body)) {
        const {
          processors
        } = pipelineDefinition;
        for (const processor of processors) {
          var _processor$inference;
          const id = (_processor$inference = processor.inference) === null || _processor$inference === void 0 ? void 0 : _processor$inference.model_id;
          if (modelIdsMap.has(id)) {
            const obj = modelIdsMap.get(id);
            if (obj === null) {
              modelIdsMap.set(id, {
                [pipelineName]: pipelineDefinition
              });
            } else {
              obj[pipelineName] = pipelineDefinition;
            }
          }
        }
      }
    } catch (error) {
      if (error.statusCode === 404) {
        // ES returns 404 when there are no pipelines
        // Instead, we should return the modelIdsMap and a 200
        return modelIdsMap;
      }
      throw error;
    }
    return modelIdsMap;
  }

  /**
   * Retrieves the network map and metadata of model ids, pipelines, and indices that are tied to the model ids.
   * @param modelIds - Array of models ids and model aliases.
   */
  async getModelsPipelinesAndIndicesMap(modelId, {
    withIndices
  }) {
    const result = {
      ingestPipelines: new Map(),
      indices: [],
      elements: [],
      details: {},
      error: null
    };
    let pipelinesResponse;
    let indicesSettings;
    try {
      var _pipelinesResponse;
      pipelinesResponse = await this.getModelsPipelines([modelId]);

      // 1. Get list of pipelines that are related to the model
      const pipelines = (_pipelinesResponse = pipelinesResponse) === null || _pipelinesResponse === void 0 ? void 0 : _pipelinesResponse.get(modelId);
      const modelNodeId = this.getNodeId(modelId, _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.TRAINED_MODEL);
      if (pipelines) {
        const pipelineIds = new Set(Object.keys(pipelines));
        result.ingestPipelines = pipelinesResponse;
        for (const pipelineId of pipelineIds) {
          const pipelineNodeId = this.getNodeId(pipelineId, _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INGEST_PIPELINE);
          result.details[pipelineNodeId] = pipelines[pipelineId];
          result.elements.push({
            data: {
              id: pipelineNodeId,
              label: pipelineId,
              type: _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INGEST_PIPELINE
            }
          });
          result.elements.push({
            data: {
              id: `${modelNodeId}~${pipelineNodeId}`,
              source: modelNodeId,
              target: pipelineNodeId
            }
          });
        }
        if (withIndices === true) {
          const pipelineIdsToDestinationIndices = {};
          let indicesPermissions;
          try {
            indicesSettings = await this._client.asInternalUser.indices.getSettings();
            const hasPrivilegesResponse = await this._client.asCurrentUser.security.hasPrivileges({
              index: [{
                names: Object.keys(indicesSettings),
                privileges: ['read']
              }]
            });
            indicesPermissions = hasPrivilegesResponse.index;
          } catch (e) {
            var _e$meta3;
            // Possible that the user doesn't have permissions to view
            // If so, gracefully exit
            if (((_e$meta3 = e.meta) === null || _e$meta3 === void 0 ? void 0 : _e$meta3.statusCode) !== 403) {
              // eslint-disable-next-line no-console
              console.error(e);
            }
            return result;
          }

          // 2. From list of model pipelines, find all indices that have pipeline set as index.default_pipeline
          for (const [indexName, {
            settings
          }] of Object.entries(indicesSettings)) {
            var _settings$index, _indicesPermissions$i;
            if (settings !== null && settings !== void 0 && (_settings$index = settings.index) !== null && _settings$index !== void 0 && _settings$index.default_pipeline && pipelineIds.has(settings.index.default_pipeline) && ((_indicesPermissions$i = indicesPermissions[indexName]) === null || _indicesPermissions$i === void 0 ? void 0 : _indicesPermissions$i.read) === true) {
              if (Array.isArray(pipelineIdsToDestinationIndices[settings.index.default_pipeline])) {
                pipelineIdsToDestinationIndices[settings.index.default_pipeline].push(indexName);
              } else {
                pipelineIdsToDestinationIndices[settings.index.default_pipeline] = [indexName];
              }
            }
          }

          // 3. Grab index information for all the indices found, and add their info to the map
          for (const [pipelineId, indexIds] of Object.entries(pipelineIdsToDestinationIndices)) {
            const pipelineNodeId = this.getNodeId(pipelineId, _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INGEST_PIPELINE);
            for (const destinationIndexId of indexIds) {
              const destinationIndexNodeId = this.getNodeId(destinationIndexId, _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX);
              const destinationIndexDetails = await this.getIndexData(destinationIndexId);
              result.indices.push(destinationIndexDetails);
              result.details[destinationIndexNodeId] = {
                ...destinationIndexDetails,
                ml_inference_models: [modelId]
              };
              result.elements.push({
                data: {
                  id: destinationIndexNodeId,
                  label: destinationIndexId,
                  type: _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX
                }
              });
              result.elements.push({
                data: {
                  id: `${pipelineNodeId}~${destinationIndexNodeId}`,
                  source: pipelineNodeId,
                  target: destinationIndexNodeId
                }
              });
            }
          }
          const destinationIndices = (0, _lodash.flatten)(Object.values(pipelineIdsToDestinationIndices));

          // 4. From these destination indices, check if there's any transforms that have the indexId as the source destination index
          if (destinationIndices.length > 0) {
            const transforms = await this.initTransformData();
            if (!transforms) return result;
            for (const destinationIndex of destinationIndices) {
              const destinationIndexNodeId = `${destinationIndex}-${_mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX}`;
              const foundTransform = transforms === null || transforms === void 0 ? void 0 : transforms.find(t => {
                const transformSourceIndex = Array.isArray(t.source.index) ? t.source.index[0] : t.source.index;
                return transformSourceIndex === destinationIndex;
              });

              // 5. If any of the transforms use these indices as source , find the destination indices to complete the map
              if (foundTransform) {
                const transformDestIndex = foundTransform.dest.index;
                const transformNodeId = `${foundTransform.id}-${_mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.TRANSFORM}`;
                const transformDestIndexNodeId = `${transformDestIndex}-${_mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX}`;
                const destIndex = await this.getIndexData(transformDestIndex);
                result.indices.push(destIndex);
                result.details[transformNodeId] = foundTransform;
                result.details[transformDestIndexNodeId] = destIndex;
                result.elements.push({
                  data: {
                    id: transformNodeId,
                    label: foundTransform.id,
                    type: _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.TRANSFORM
                  }
                }, {
                  data: {
                    id: transformDestIndexNodeId,
                    label: transformDestIndex,
                    type: _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.INDEX
                  }
                });
                result.elements.push({
                  data: {
                    id: `${destinationIndexNodeId}~${transformNodeId}`,
                    source: destinationIndexNodeId,
                    target: transformNodeId
                  }
                }, {
                  data: {
                    id: `${transformNodeId}~${transformDestIndexNodeId}`,
                    source: transformNodeId,
                    target: transformDestIndexNodeId
                  }
                });
              }
            }
          }
        }
      }
      return result;
    } catch (error) {
      if (error.statusCode === 404) {
        // ES returns 404 when there are no pipelines
        // Instead, we should return the modelIdsMap and a 200
        return result;
      }
      throw error;
    }
  }

  /**
   * Deletes associated pipelines of the requested model
   * @param modelIds
   */
  async deleteModelPipelines(modelIds) {
    const pipelines = await this.getModelsPipelines(modelIds);
    const pipelinesIds = [...new Set([...pipelines.values()].flatMap(v => Object.keys(v)))];
    await Promise.all(pipelinesIds.map(id => this._client.asCurrentUser.ingest.deletePipeline({
      id
    })));
  }

  /**
   * Returns a list of elastic curated models available for download.
   */
  async getModelDownloads() {
    var _this$_cloud;
    // We assume that ML nodes in Cloud are always on linux-x86_64, even if other node types aren't.
    const isCloud = !!((_this$_cloud = this._cloud) !== null && _this$_cloud !== void 0 && _this$_cloud.cloudId);
    const nodesInfoResponse = await this._client.asInternalUser.transport.request({
      method: 'GET',
      path: `/_nodes/ml:true/os`
    });
    let osName;
    let arch;
    // Indicates that all ML nodes have the same architecture
    let sameArch = true;
    for (const node of Object.values(nodesInfoResponse.nodes)) {
      var _node$os3, _node$os4;
      if (!osName) {
        var _node$os;
        osName = (_node$os = node.os) === null || _node$os === void 0 ? void 0 : _node$os.name;
      }
      if (!arch) {
        var _node$os2;
        arch = (_node$os2 = node.os) === null || _node$os2 === void 0 ? void 0 : _node$os2.arch;
      }
      if (((_node$os3 = node.os) === null || _node$os3 === void 0 ? void 0 : _node$os3.name) !== osName || ((_node$os4 = node.os) === null || _node$os4 === void 0 ? void 0 : _node$os4.arch) !== arch) {
        sameArch = false;
        break;
      }
    }
    const modelDefinitionMap = new Map();
    for (const [modelId, def] of Object.entries(_mlTrainedModelsUtils.ELASTIC_MODEL_DEFINITIONS)) {
      const recommended = isCloud && def.os === 'Linux' && def.arch === 'amd64' || sameArch && !!(def !== null && def !== void 0 && def.os) && (def === null || def === void 0 ? void 0 : def.os) === osName && (def === null || def === void 0 ? void 0 : def.arch) === arch;
      const {
        modelName
      } = def;
      const modelDefinitionResponse = {
        ...def,
        ...(recommended ? {
          recommended
        } : {}),
        supported: !!def.default || recommended,
        model_id: modelId
      };
      if (modelDefinitionMap.has(modelName)) {
        modelDefinitionMap.get(modelName).push(modelDefinitionResponse);
      } else {
        modelDefinitionMap.set(modelName, [modelDefinitionResponse]);
      }
    }

    // check if there is no recommended, so we mark default as recommended
    for (const arr of modelDefinitionMap.values()) {
      const defaultModel = arr.find(a => a.default);
      const recommendedModel = arr.find(a => a.recommended);
      if (defaultModel && !recommendedModel) {
        delete defaultModel.default;
        defaultModel.recommended = true;
      }
    }
    return [...modelDefinitionMap.values()].flat();
  }

  /**
   * Provides an appropriate model ID and configuration for download based on the current cluster architecture.
   *
   * @param modelName
   * @param options
   * @returns
   */
  async getCuratedModelConfig(modelName, options) {
    const modelDownloadConfig = (await this.getModelDownloads()).filter(model => model.modelName === modelName);
    let requestedModel;
    let recommendedModel;
    let defaultModel;
    for (const model of modelDownloadConfig) {
      if ((options === null || options === void 0 ? void 0 : options.version) === model.version) {
        requestedModel = model;
        if (model.recommended) {
          requestedModel = model;
          break;
        }
      } else if (model.recommended) {
        recommendedModel = model;
      } else if (model.default) {
        defaultModel = model;
      }
    }
    if (!requestedModel && !defaultModel && !recommendedModel) {
      throw new Error('Requested model not found');
    }
    return requestedModel || recommendedModel || defaultModel;
  }

  /**
   * Provides an ELSER model name and configuration for download based on the current cluster architecture.
   * The current default version is 2. If running on Cloud it returns the Linux x86_64 optimized version.
   * If any of the ML nodes run a different OS rather than Linux, or the CPU architecture isn't x86_64,
   * a portable version of the model is returned.
   */
  async getELSER(options) {
    return await this.getCuratedModelConfig('elser', options);
  }

  /**
   * Puts the requested ELSER model into elasticsearch, triggering elasticsearch to download the model.
   * Assigns the model to the * space.
   * @param modelId
   * @param mlSavedObjectService
   */
  async installElasticModel(modelId, mlSavedObjectService) {
    const availableModels = await this.getModelDownloads();
    const model = availableModels.find(m => m.model_id === modelId);
    if (!model) {
      throw _boom.default.notFound('Model not found');
    }
    let esModelExists = false;
    try {
      await this._client.asInternalUser.ml.getTrainedModels({
        model_id: modelId
      });
      esModelExists = true;
    } catch (error) {
      if (error.statusCode !== 404) {
        throw error;
      }
      // model doesn't exist, ignore error
    }
    if (esModelExists) {
      throw _boom.default.badRequest('Model already exists');
    }
    const putResponse = await this._mlClient.putTrainedModel({
      model_id: model.model_id,
      body: model.config
    });
    await mlSavedObjectService.updateTrainedModelsSpaces([modelId], ['*'], []);
    return putResponse;
  }
  /**
   * Puts the requested Inference endpoint id into elasticsearch, triggering elasticsearch to create the inference endpoint id
   * @param inferenceId - Inference Endpoint Id
   * @param taskType - Inference Task type. Either sparse_embedding or text_embedding
   * @param inferenceConfig - Model configuration based on service type
   */
  async createInferenceEndpoint(inferenceId, taskType, inferenceConfig) {
    try {
      const result = await this._client.asCurrentUser.inference.put({
        inference_id: inferenceId,
        task_type: taskType,
        inference_config: inferenceConfig
      }, {
        maxRetries: 0
      });
      return result;
    } catch (error) {
      // Request timeouts will usually occur when the model is being downloaded/deployed
      // Erroring out is misleading in these cases, so we return the model_id and task_type
      if (error.name === 'TimeoutError') {
        return {
          model_id: inferenceConfig.service,
          task_type: taskType
        };
      } else {
        throw error;
      }
    }
  }
  async getModelsDownloadStatus() {
    var _result$tasks;
    const result = await this._client.asInternalUser.tasks.list({
      actions: 'xpack/ml/model_import[n]',
      detailed: true,
      group_by: 'none'
    });
    if (!((_result$tasks = result.tasks) !== null && _result$tasks !== void 0 && _result$tasks.length)) {
      return {};
    }

    // Groups results by model id
    const byModelId = result.tasks.reduce((acc, task) => {
      const modelId = task.description.replace(`model_id-`, '');
      acc[modelId] = {
        downloaded_parts: task.status.downloaded_parts,
        total_parts: task.status.total_parts
      };
      return acc;
    }, {});
    return byModelId;
  }
}
exports.ModelsProvider = ModelsProvider;