"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QuickJobCreatorBase = void 0;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _presentationPublishing = require("@kbn/presentation-publishing");
var _esQuery = require("@kbn/es-query");
var _job_utils = require("../../../../../common/util/job_utils");
var _new_job_utils = require("../utils/new_job_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function mergeQueriesCheck(objValue, srcValue) {
  if (Array.isArray(objValue)) {
    const combinedQuery = objValue.concat(srcValue);
    return (0, _lodash.uniqWith)(combinedQuery, _lodash.isEqual);
  }
}
class QuickJobCreatorBase {
  constructor(dataViews, kibanaConfig, timeFilter, dashboardService, mlApi) {
    this.dataViews = dataViews;
    this.kibanaConfig = kibanaConfig;
    this.timeFilter = timeFilter;
    this.dashboardService = dashboardService;
    this.mlApi = mlApi;
  }
  async putJobAndDataFeed({
    jobId,
    datafeedConfig,
    jobConfig,
    createdByLabel,
    start,
    end,
    startJob,
    runInRealTime,
    dashboard
  }) {
    const datafeedId = (0, _job_utils.createDatafeedId)(jobId);
    const datafeed = {
      ...datafeedConfig,
      job_id: jobId,
      datafeed_id: datafeedId
    };
    const job = {
      ...jobConfig,
      job_id: jobId,
      custom_settings: {
        created_by: createdByLabel,
        ...(dashboard ? await this.getCustomUrls(dashboard, datafeed) : {})
      }
    };
    const result = {
      jobCreated: {
        success: false
      },
      datafeedCreated: {
        success: false
      },
      jobOpened: {
        success: false
      },
      datafeedStarted: {
        success: false
      }
    };

    // calculate model memory limit
    try {
      if (start !== undefined && end !== undefined && job.data_description.time_field !== undefined && datafeedConfig.indices.length > 0) {
        const {
          modelMemoryLimit
        } = await (0, _rxjs.firstValueFrom)(this.mlApi.calculateModelMemoryLimit$({
          datafeedConfig: datafeed,
          analysisConfig: job.analysis_config,
          indexPattern: datafeedConfig.indices[0],
          query: datafeedConfig.query,
          timeFieldName: job.data_description.time_field,
          earliestMs: start,
          latestMs: end
        }));
        if (job.analysis_limits === undefined) {
          job.analysis_limits = {};
        }
        job.analysis_limits.model_memory_limit = modelMemoryLimit;
      }
    } catch (error) {
      // could not calculate mml, continue with job creation as default value will be used.
      // eslint-disable-next-line no-console
      console.error('could not calculate model memory limit', error);
    }

    // put job
    try {
      await this.mlApi.addJob({
        jobId: job.job_id,
        job
      });
    } catch (error) {
      result.jobCreated.error = error;
      return result;
    }
    result.jobCreated.success = true;

    // put datafeed
    try {
      await this.mlApi.addDatafeed({
        datafeedId,
        datafeedConfig: datafeed
      });
    } catch (error) {
      result.datafeedCreated.error = error;
      return result;
    }
    result.datafeedCreated.success = true;
    if (startJob) {
      // open job, ignore error if already open
      try {
        await this.mlApi.openJob({
          jobId
        });
      } catch (error) {
        // job may already be open, so ignore 409 error.
        if (error.body.statusCode !== 409) {
          result.jobOpened.error = error;
          return result;
        }
      }
      result.jobOpened.success = true;

      // start datafeed
      try {
        await this.mlApi.startDatafeed({
          datafeedId,
          start,
          ...(runInRealTime ? {} : {
            end
          })
        });
      } catch (error) {
        result.datafeedStarted.error = error;
        return result;
      }
      result.datafeedStarted.success = true;
    }
    return result;
  }
  combineQueriesAndFilters(dashboard, vis, dataView, layerQuery) {
    let mergedVisAndLayerQueries;
    const {
      combinedQuery: dashboardQueries
    } = (0, _new_job_utils.createQueries)({
      query: dashboard.query,
      filter: dashboard.filters
    }, dataView, this.kibanaConfig);
    const {
      combinedQuery: visQueries
    } = (0, _new_job_utils.createQueries)({
      query: vis.query,
      filter: vis.filters
    }, dataView, this.kibanaConfig);
    if (layerQuery) {
      const {
        combinedQuery: layerQueries
      } = (0, _new_job_utils.createQueries)({
        query: layerQuery.query,
        filter: layerQuery.filters
      }, dataView, this.kibanaConfig);
      // combine vis and layer queries if layer-level query exists
      mergedVisAndLayerQueries = (0, _lodash.mergeWith)(visQueries, layerQueries, mergeQueriesCheck);
    }
    const mergedQueries = (0, _lodash.mergeWith)(dashboardQueries, mergedVisAndLayerQueries ? mergedVisAndLayerQueries : visQueries, mergeQueriesCheck);
    return mergedQueries;
  }
  async createDashboardLink(dashboard, datafeedConfig) {
    var _dashboard$savedObjec, _this$dashboardServic, _getPanelTitle;
    const savedObjectId = (_dashboard$savedObjec = dashboard.savedObjectId) === null || _dashboard$savedObjec === void 0 ? void 0 : _dashboard$savedObjec.value;
    if (!savedObjectId) {
      return null;
    }
    const params = {
      dashboardId: savedObjectId,
      timeRange: {
        from: '$earliest$',
        to: '$latest$',
        mode: 'absolute'
      },
      filters: (0, _job_utils.getFiltersForDSLQuery)(datafeedConfig.query, undefined, datafeedConfig.job_id, _esQuery.FilterStateStore.GLOBAL_STATE)
    };
    const location = await ((_this$dashboardServic = this.dashboardService.locator) === null || _this$dashboardServic === void 0 ? void 0 : _this$dashboardServic.getLocation(params));
    if (location === undefined) {
      return null;
    }
    const url = `${location.app}${location.path}`;
    const urlName = _i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.namedUrlDashboard', {
      defaultMessage: 'Open {dashboardTitle}',
      values: {
        dashboardTitle: (_getPanelTitle = (0, _presentationPublishing.getPanelTitle)(dashboard)) !== null && _getPanelTitle !== void 0 ? _getPanelTitle : 'dashboard'
      }
    });
    return {
      url_name: urlName,
      url_value: url,
      time_range: 'auto'
    };
  }
  async getCustomUrls(dashboard, datafeedConfig) {
    const customUrls = await this.createDashboardLink(dashboard, datafeedConfig);
    return dashboard !== undefined && customUrls !== null ? {
      custom_urls: [customUrls]
    } : {};
  }
}
exports.QuickJobCreatorBase = QuickJobCreatorBase;