"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAnalyticsJobsStats = getAnalyticsJobsStats;
exports.getInitialAnalyticsStats = getInitialAnalyticsStats;
exports.useGetAnalytics = exports.isGetDataFrameAnalyticsStatsResponseOk = void 0;
var _i18n = require("@kbn/i18n");
var _mlDataFrameAnalyticsUtils = require("@kbn/ml-data-frame-analytics-utils");
var _kibana = require("../../../../../contexts/kibana");
var _common = require("../../../../common");
var _common2 = require("../../components/analytics_list/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isGetDataFrameAnalyticsStatsResponseOk = arg => {
  return {}.hasOwnProperty.call(arg, 'count') && {}.hasOwnProperty.call(arg, 'data_frame_analytics') && Array.isArray(arg.data_frame_analytics);
};
exports.isGetDataFrameAnalyticsStatsResponseOk = isGetDataFrameAnalyticsStatsResponseOk;
/**
 * Gets initial object for analytics stats.
 */
function getInitialAnalyticsStats() {
  return {
    total: {
      label: _i18n.i18n.translate('xpack.ml.overview.statsBar.totalAnalyticsLabel', {
        defaultMessage: 'Total'
      }),
      value: 0,
      show: true
    },
    started: {
      label: _i18n.i18n.translate('xpack.ml.overview.statsBar.runningAnalyticsLabel', {
        defaultMessage: 'Running'
      }),
      value: 0,
      show: true
    },
    stopped: {
      label: _i18n.i18n.translate('xpack.ml.overview.statsBar.stoppedAnalyticsLabel', {
        defaultMessage: 'Stopped'
      }),
      value: 0,
      show: true
    },
    failed: {
      label: _i18n.i18n.translate('xpack.ml.overview.statsBar.failedAnalyticsLabel', {
        defaultMessage: 'Failed'
      }),
      value: 0,
      show: false
    }
  };
}

/**
 * Gets analytics jobs stats formatted for the stats bar.
 */
function getAnalyticsJobsStats(analyticsStats) {
  const resultStats = analyticsStats.data_frame_analytics.reduce((acc, {
    state
  }) => {
    if ((0, _common2.isDataFrameAnalyticsFailed)(state)) {
      acc.failed.value = ++acc.failed.value;
    } else if ((0, _common2.isDataFrameAnalyticsRunning)(state)) {
      acc.started.value = ++acc.started.value;
    } else if ((0, _common2.isDataFrameAnalyticsStopped)(state)) {
      acc.stopped.value = ++acc.stopped.value;
    }
    return acc;
  }, getInitialAnalyticsStats());
  resultStats.failed.show = resultStats.failed.value > 0;
  resultStats.total.value = analyticsStats.count;
  if (resultStats.total.value === 0) {
    resultStats.started.show = false;
    resultStats.stopped.show = false;
  }
  return resultStats;
}
const useGetAnalytics = (setAnalytics, setAnalyticsStats, setErrorMessage, setIsInitialized, setJobsAwaitingNodeCount, blockRefresh) => {
  const mlApi = (0, _kibana.useMlApi)();
  let concurrentLoads = 0;
  const getAnalytics = async (forceRefresh = false) => {
    if (forceRefresh === true || blockRefresh === false) {
      _common.refreshAnalyticsList$.next(_common.REFRESH_ANALYTICS_LIST_STATE.LOADING);
      concurrentLoads++;
      if (concurrentLoads > 1) {
        return;
      }
      try {
        const analyticsConfigs = await mlApi.dataFrameAnalytics.getDataFrameAnalytics();
        const analyticsStats = await mlApi.dataFrameAnalytics.getDataFrameAnalyticsStats();
        const analyticsStatsResult = isGetDataFrameAnalyticsStatsResponseOk(analyticsStats) ? getAnalyticsJobsStats(analyticsStats) : undefined;
        let jobsAwaitingNodeCount = 0;
        const tableRows = analyticsConfigs.data_frame_analytics.reduce((reducedtableRows, config) => {
          const stats = isGetDataFrameAnalyticsStatsResponseOk(analyticsStats) ? analyticsStats.data_frame_analytics.find(d => config.id === d.id) : undefined;

          // A newly created analytics job might not have corresponding stats yet.
          // If that's the case we just skip the job and don't add it to the analytics jobs list yet.
          if (!(0, _common2.isDataFrameAnalyticsStats)(stats)) {
            return reducedtableRows;
          }
          if (stats.state === _mlDataFrameAnalyticsUtils.DATA_FRAME_TASK_STATE.STARTING && stats.node === undefined) {
            jobsAwaitingNodeCount++;
          }

          // Table with expandable rows requires `id` on the outer most level
          reducedtableRows.push({
            checkpointing: {},
            config,
            id: config.id,
            job_type: (0, _mlDataFrameAnalyticsUtils.getAnalysisType)(config.analysis),
            mode: _common2.DATA_FRAME_MODE.BATCH,
            state: stats.state,
            stats
          });
          return reducedtableRows;
        }, []);
        setAnalytics(tableRows);
        setAnalyticsStats(analyticsStatsResult);
        setErrorMessage(undefined);
        setIsInitialized(true);
        setJobsAwaitingNodeCount(jobsAwaitingNodeCount);
        _common.refreshAnalyticsList$.next(_common.REFRESH_ANALYTICS_LIST_STATE.IDLE);
      } catch (e) {
        // An error is followed immediately by setting the state to idle.
        // This way we're able to treat ERROR as a one-time-event like REFRESH.
        _common.refreshAnalyticsList$.next(_common.REFRESH_ANALYTICS_LIST_STATE.ERROR);
        _common.refreshAnalyticsList$.next(_common.REFRESH_ANALYTICS_LIST_STATE.IDLE);
        setAnalytics([]);
        setAnalyticsStats(undefined);
        setErrorMessage(e);
        setIsInitialized(true);
      }
      concurrentLoads--;
      if (concurrentLoads > 0) {
        concurrentLoads = 0;
        getAnalytics(true);
        return;
      }
    }
  };
  return getAnalytics;
};
exports.useGetAnalytics = useGetAnalytics;