"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importListItemRoute = void 0;
var _path = require("path");
var _configSchema = require("@kbn/config-schema");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _zodHelpers = require("@kbn/zod-helpers");
var _api = require("@kbn/securitysolution-lists-common/api");
var _utils = require("../utils");
var _create_stream_from_buffer = require("../utils/create_stream_from_buffer");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validFileExtensions = ['.csv', '.txt'];
const importListItemRoute = (router, config) => {
  router.versioned.post({
    access: 'public',
    options: {
      body: {
        accepts: ['multipart/form-data'],
        maxBytes: config.maxImportPayloadBytes,
        parse: false
      },
      tags: ['access:lists-all'],
      timeout: {
        payload: config.importTimeout.asMilliseconds()
      }
    },
    path: `${_securitysolutionListConstants.LIST_ITEM_URL}/_import`
  }).addVersion({
    validate: {
      request: {
        body: _configSchema.schema.buffer(),
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_api.ImportListItemsRequestQuery)
      }
    },
    version: '2023-10-31'
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const {
        deserializer,
        list_id: listId,
        serializer,
        type,
        refresh
      } = request.query;
      const lists = await (0, _.getListClient)(context);
      const filename = await lists.getImportFilename({
        stream: (0, _create_stream_from_buffer.createStreamFromBuffer)(request.body)
      });
      if (!filename) {
        return siemResponse.error({
          body: 'To import a list item, the file name must be specified',
          statusCode: 400
        });
      }
      const fileExtension = (0, _path.extname)(filename).toLowerCase();
      if (!validFileExtensions.includes(fileExtension)) {
        return siemResponse.error({
          body: `Unsupported media type. File must be one of the following types: [${validFileExtensions.join(', ')}]`,
          statusCode: 415
        });
      }
      const stream = (0, _create_stream_from_buffer.createStreamFromBuffer)(request.body);
      const listDataExists = await lists.getListDataStreamExists();
      if (!listDataExists) {
        const listIndexExists = await lists.getListIndexExists();
        if (!listIndexExists) {
          return siemResponse.error({
            body: `To import a list item, the data steam must exist first. Data stream "${lists.getListName()}" does not exist`,
            statusCode: 400
          });
        }
        // otherwise migration is needed
        await lists.migrateListIndexToDataStream();
      }
      const listItemDataExists = await lists.getListItemDataStreamExists();
      if (!listItemDataExists) {
        const listItemIndexExists = await lists.getListItemIndexExists();
        if (!listItemIndexExists) {
          return siemResponse.error({
            body: `To import a list item, the data steam must exist first. Data stream "${lists.getListItemName()}" does not exist`,
            statusCode: 400
          });
        }
        // otherwise migration is needed
        await lists.migrateListItemIndexToDataStream();
      }
      if (listId != null) {
        const list = await lists.getList({
          id: listId
        });
        if (list == null) {
          return siemResponse.error({
            body: `list id: "${listId}" does not exist`,
            statusCode: 409
          });
        }
        await lists.importListItemsToStream({
          deserializer: list.deserializer,
          listId,
          meta: undefined,
          refresh,
          serializer: list.serializer,
          stream,
          type: list.type,
          version: 1
        });
        return response.ok({
          body: _api.ImportListItemsResponse.parse(list)
        });
      } else if (type != null) {
        const importedList = await lists.importListItemsToStream({
          deserializer,
          listId: undefined,
          meta: undefined,
          refresh,
          serializer,
          stream,
          type,
          version: 1
        });
        if (importedList == null) {
          return siemResponse.error({
            body: 'Unable to parse a valid fileName during import',
            statusCode: 400
          });
        }
        return response.ok({
          body: _api.ImportListItemsResponse.parse(importedList)
        });
      } else {
        return siemResponse.error({
          body: 'Either type or list_id need to be defined in the query',
          statusCode: 400
        });
      }
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.importListItemRoute = importListItemRoute;