"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.duplicateExceptionsRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _zodHelpers = require("@kbn/zod-helpers");
var _api = require("@kbn/securitysolution-exceptions-common/api");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const duplicateExceptionsRoute = router => {
  router.versioned.post({
    access: 'public',
    options: {
      tags: ['access:lists-all']
    },
    path: `${_securitysolutionListConstants.EXCEPTION_LIST_URL}/_duplicate`
  }).addVersion({
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_api.DuplicateExceptionListRequestQuery)
      }
    },
    version: '2023-10-31'
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const {
        list_id: listId,
        namespace_type: namespaceType,
        include_expired_exceptions: includeExpiredExceptionsString
      } = request.query;
      const exceptionListsClient = await (0, _utils.getExceptionListClient)(context);

      // fetch list container
      const listToDuplicate = await exceptionListsClient.getExceptionList({
        id: undefined,
        listId,
        namespaceType
      });
      if (listToDuplicate == null) {
        return siemResponse.error({
          body: `exception list id: "${listId}" does not exist`,
          statusCode: 404
        });
      }

      // Defaults to including expired exceptions if query param is not present
      const includeExpiredExceptions = includeExpiredExceptionsString !== undefined ? includeExpiredExceptionsString === 'true' : true;
      const duplicatedList = await exceptionListsClient.duplicateExceptionListAndItems({
        includeExpiredExceptions,
        list: listToDuplicate,
        namespaceType
      });
      if (duplicatedList == null) {
        return siemResponse.error({
          body: `unable to duplicate exception list with list_id: ${listId} - action not allowed`,
          statusCode: 405
        });
      }
      return response.ok({
        body: _api.DuplicateExceptionListResponse.parse(duplicatedList)
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.duplicateExceptionsRoute = duplicateExceptionsRoute;