"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createExceptionListItemRoute = void 0;
var _uuid = require("uuid");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _zodHelpers = require("@kbn/zod-helpers");
var _api = require("@kbn/securitysolution-exceptions-common/api");
var _utils = require("./utils");
var _get_exception_list_client = require("./utils/get_exception_list_client");
var _endpoint_disallowed_fields = require("./endpoint_disallowed_fields");
var _validate = require("./validate");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createExceptionListItemRoute = router => {
  router.versioned.post({
    access: 'public',
    options: {
      tags: ['access:lists-all']
    },
    path: _securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL
  }).addVersion({
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_api.CreateExceptionListItemRequestBody)
      }
    },
    version: '2023-10-31'
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const {
        namespace_type: namespaceType,
        name,
        tags,
        meta,
        comments,
        description,
        entries,
        item_id: itemId = (0, _uuid.v4)(),
        list_id: listId,
        os_types: osTypes,
        type,
        expire_time: expireTime
      } = request.body;
      const exceptionLists = await (0, _get_exception_list_client.getExceptionListClient)(context);
      const exceptionList = await exceptionLists.getExceptionList({
        id: undefined,
        listId,
        namespaceType
      });
      if (exceptionList == null) {
        return siemResponse.error({
          body: `exception list id: "${listId}" does not exist`,
          statusCode: 404
        });
      }
      const exceptionListItem = await exceptionLists.getExceptionListItem({
        id: undefined,
        itemId,
        namespaceType
      });
      if (exceptionListItem != null) {
        return siemResponse.error({
          body: `exception list item id: "${itemId}" already exists`,
          statusCode: 409
        });
      }
      if (exceptionList.type === 'endpoint') {
        const error = (0, _validate.validateEndpointExceptionItemEntries)(request.body.entries);
        if (error != null) {
          return siemResponse.error(error);
        }
        for (const entry of entries) {
          if (_endpoint_disallowed_fields.endpointDisallowedFields.includes(entry.field)) {
            return siemResponse.error({
              body: `cannot add endpoint exception item on field ${entry.field}`,
              statusCode: 400
            });
          }
        }
      }
      const createdListItem = await exceptionLists.createExceptionListItem({
        comments,
        description,
        entries,
        expireTime,
        itemId,
        listId,
        meta,
        name,
        namespaceType,
        osTypes,
        tags,
        type
      });
      const listSizeError = await (0, _validate.validateExceptionListSize)(exceptionLists, listId, namespaceType);
      if (listSizeError != null) {
        await exceptionLists.deleteExceptionListItemById({
          id: createdListItem.id,
          namespaceType
        });
        return siemResponse.error(listSizeError);
      }
      return response.ok({
        body: _api.CreateExceptionListItemResponse.parse(createdListItem)
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.createExceptionListItemRoute = createExceptionListItemRoute;