"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToPersistable = convertToPersistable;
exports.convertToRuntime = convertToRuntime;
exports.isPersistedLinkedByValueAnnotationsLayer = exports.isPersistedByValueAnnotationsLayer = exports.isPersistedByReferenceAnnotationsLayer = exports.isPersistedAnnotationsLayer = void 0;
var _uuid = require("uuid");
var _eventAnnotationCommon = require("@kbn/event-annotation-common");
var _lodash = require("lodash");
var _charts = require("@elastic/charts");
var _layer_types = require("../../../common/layer_types");
var _visualization_helpers = require("./visualization_helpers");
var _utils = require("../../utils");
var _state_helpers = require("./state_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isPersistedByReferenceAnnotationsLayer = layer => isPersistedAnnotationsLayer(layer) && layer.persistanceType === 'byReference';
exports.isPersistedByReferenceAnnotationsLayer = isPersistedByReferenceAnnotationsLayer;
const isPersistedLinkedByValueAnnotationsLayer = layer => isPersistedAnnotationsLayer(layer) && layer.persistanceType === 'linked';

/**
 * This is the type of hybrid layer we get after the user has made a change to
 * a by-reference annotation layer and saved the visualization without
 * first saving the changes to the library annotation layer.
 *
 * We maintain the link to the library annotation group, but allow the users
 * changes (persisted in the visualization state) to override the attributes in
 * the library version until the user
 * - saves the changes to the library annotation group
 * - reverts the changes
 * - unlinks the layer from the library annotation group
 */

// props made optional for backwards compatibility since this is how the existing saved objects are
exports.isPersistedLinkedByValueAnnotationsLayer = isPersistedLinkedByValueAnnotationsLayer;
function convertToRuntime(state, annotationGroups, references) {
  let newState = (0, _lodash.cloneDeep)(injectReferences(state, annotationGroups, references));
  newState = convertToLegendStats(newState);
  return newState;
}
function convertToPersistable(state) {
  const persistableState = state;
  const savedObjectReferences = [];
  const persistableLayers = [];
  persistableState.layers.forEach(layer => {
    if (!(0, _visualization_helpers.isAnnotationsLayer)(layer)) {
      persistableLayers.push(layer);
    } else {
      if ((0, _visualization_helpers.isByReferenceAnnotationsLayer)(layer)) {
        const referenceName = `ref-${(0, _uuid.v4)()}`;
        savedObjectReferences.push({
          type: _eventAnnotationCommon.EVENT_ANNOTATION_GROUP_TYPE,
          id: layer.annotationGroupId,
          name: referenceName
        });
        if (!(0, _state_helpers.annotationLayerHasUnsavedChanges)(layer)) {
          const persistableLayer = {
            persistanceType: 'byReference',
            layerId: layer.layerId,
            layerType: layer.layerType,
            annotationGroupRef: referenceName
          };
          persistableLayers.push(persistableLayer);
        } else {
          const persistableLayer = {
            persistanceType: 'linked',
            cachedMetadata: layer.cachedMetadata || {
              title: layer.__lastSaved.title,
              description: layer.__lastSaved.description,
              tags: layer.__lastSaved.tags
            },
            layerId: layer.layerId,
            layerType: layer.layerType,
            annotationGroupRef: referenceName,
            annotations: layer.annotations,
            ignoreGlobalFilters: layer.ignoreGlobalFilters
          };
          persistableLayers.push(persistableLayer);
          savedObjectReferences.push({
            type: 'index-pattern',
            id: layer.indexPatternId,
            name: getLayerReferenceName(layer.layerId)
          });
        }
      } else {
        const {
          indexPatternId,
          ...persistableLayer
        } = layer;
        savedObjectReferences.push({
          type: 'index-pattern',
          id: indexPatternId,
          name: getLayerReferenceName(layer.layerId)
        });
        persistableLayers.push({
          ...persistableLayer,
          persistanceType: 'byValue'
        });
      }
    }
  });
  return {
    savedObjectReferences,
    state: {
      ...persistableState,
      layers: persistableLayers
    }
  };
}
const isPersistedAnnotationsLayer = layer => layer.layerType === _layer_types.layerTypes.ANNOTATIONS && !('indexPatternId' in layer);
exports.isPersistedAnnotationsLayer = isPersistedAnnotationsLayer;
const isPersistedByValueAnnotationsLayer = layer => isPersistedAnnotationsLayer(layer) && (layer.persistanceType === 'byValue' || !layer.persistanceType);
exports.isPersistedByValueAnnotationsLayer = isPersistedByValueAnnotationsLayer;
function getLayerReferenceName(layerId) {
  return `xy-visualization-layer-${layerId}`;
}
function needsInjectReferences(state) {
  return state.layers.some(isPersistedAnnotationsLayer);
}
function injectReferences(state, annotationGroups, references) {
  if (!references || !references.length) {
    return state;
  }
  if (!needsInjectReferences(state)) {
    return state;
  }
  if (!annotationGroups) {
    throw new Error('xy visualization: injecting references relies on annotation groups but they were not provided');
  }

  // called on-demand since indexPattern reference won't be here on the vis if its a by-reference group
  const getIndexPatternIdFromReferences = annotationLayerId => {
    var _references$find;
    const fallbackIndexPatternId = references.find(({
      type
    }) => type === 'index-pattern').id;
    return ((_references$find = references.find(({
      name
    }) => name === getLayerReferenceName(annotationLayerId))) === null || _references$find === void 0 ? void 0 : _references$find.id) || fallbackIndexPatternId;
  };
  return {
    ...state,
    layers: state.layers.map(persistedLayer => {
      if (!isPersistedAnnotationsLayer(persistedLayer)) {
        return persistedLayer;
      }
      let injectedLayer;
      if (isPersistedByValueAnnotationsLayer(persistedLayer)) {
        injectedLayer = {
          ...persistedLayer,
          indexPatternId: getIndexPatternIdFromReferences(persistedLayer.layerId)
        };
      } else {
        var _references$find2;
        const annotationGroupId = references === null || references === void 0 ? void 0 : (_references$find2 = references.find(({
          name
        }) => name === persistedLayer.annotationGroupRef)) === null || _references$find2 === void 0 ? void 0 : _references$find2.id;
        const annotationGroup = annotationGroupId ? annotationGroups[annotationGroupId] : undefined;
        if (!annotationGroupId || !annotationGroup) {
          return undefined; // the annotation group this layer was referencing is gone, so remove the layer
        }

        // declared as a separate variable for type checking
        const commonProps = {
          layerId: persistedLayer.layerId,
          layerType: persistedLayer.layerType,
          annotationGroupId,
          __lastSaved: annotationGroup
        };
        if (isPersistedByReferenceAnnotationsLayer(persistedLayer)) {
          // a clean by-reference layer inherits everything from the library annotation group
          injectedLayer = {
            ...commonProps,
            ignoreGlobalFilters: annotationGroup.ignoreGlobalFilters,
            indexPatternId: annotationGroup.indexPatternId,
            annotations: (0, _lodash.cloneDeep)(annotationGroup.annotations)
          };
        } else {
          // a linked by-value layer gets settings from visualization state while
          // still maintaining the reference to the library annotation group
          injectedLayer = {
            ...commonProps,
            ignoreGlobalFilters: persistedLayer.ignoreGlobalFilters,
            indexPatternId: getIndexPatternIdFromReferences(persistedLayer.layerId),
            annotations: (0, _lodash.cloneDeep)(persistedLayer.annotations),
            cachedMetadata: persistedLayer.cachedMetadata
          };
        }
      }
      return injectedLayer;
    }).filter(_utils.nonNullable)
  };
}
function convertToLegendStats(state) {
  if ('valuesInLegend' in state) {
    const valuesInLegend = state.valuesInLegend;
    delete state.valuesInLegend;
    const result = {
      ...state,
      legend: {
        ...state.legend,
        legendStats: [...new Set([...(valuesInLegend ? [_charts.LegendValue.CurrentAndLastValue] : []), ...(state.legend.legendStats || [])])]
      }
    };
    return result;
  }
  return state;
}