"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseBedrockStreamAsAsyncIterator = exports.parseBedrockStream = exports.handleBedrockChunk = void 0;
var _promises = require("stream/promises");
var _eventstreamCodec = require("@smithy/eventstream-codec");
var _utilUtf = require("@smithy/util-utf8");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const parseBedrockStreamAsAsyncIterator = async function* (responseStream, logger, abortSignal) {
  if (abortSignal) {
    abortSignal.addEventListener('abort', () => {
      responseStream.destroy(new Error('Aborted'));
    });
  }
  try {
    for await (const chunk of responseStream) {
      const bedrockChunk = handleBedrockChunk({
        chunk,
        bedrockBuffer: new Uint8Array(0),
        logger
      });
      yield bedrockChunk.decodedChunk;
    }
  } catch (err) {
    if (abortSignal !== null && abortSignal !== void 0 && abortSignal.aborted) {
      logger.info('Bedrock stream parsing was aborted.');
    } else {
      throw err;
    }
  }
};
exports.parseBedrockStreamAsAsyncIterator = parseBedrockStreamAsAsyncIterator;
const parseBedrockStream = async (responseStream, logger, abortSignal, tokenHandler) => {
  const responseBuffer = [];
  if (abortSignal) {
    abortSignal.addEventListener('abort', () => {
      responseStream.destroy(new Error('Aborted'));
      return parseBedrockBuffer(responseBuffer, logger);
    });
  }
  responseStream.on('data', chunk => {
    // special encoding for bedrock, do not attempt to convert to string
    responseBuffer.push(chunk);
    if (tokenHandler) {
      // Initialize an empty Uint8Array to store the concatenated buffer.
      const bedrockBuffer = new Uint8Array(0);
      handleBedrockChunk({
        chunk,
        bedrockBuffer,
        logger,
        chunkHandler: tokenHandler
      });
    }
  });
  await (0, _promises.finished)(responseStream).catch(err => {
    if (abortSignal !== null && abortSignal !== void 0 && abortSignal.aborted) {
      logger.info('Bedrock stream parsing was aborted.');
    } else {
      throw err;
    }
  });
  return parseBedrockBuffer(responseBuffer, logger);
};

/**
 * Parses a Bedrock buffer from an array of chunks.
 *
 * @param {Uint8Array[]} chunks - Array of Uint8Array chunks to be parsed.
 * @returns {string} - Parsed string from the Bedrock buffer.
 */
exports.parseBedrockStream = parseBedrockStream;
const parseBedrockBuffer = (chunks, logger) => {
  // Initialize an empty Uint8Array to store the concatenated buffer.
  let bedrockBuffer = new Uint8Array(0);

  // Map through each chunk to process the Bedrock buffer.
  return chunks.map(chunk => {
    const processedChunk = handleBedrockChunk({
      chunk,
      bedrockBuffer,
      logger
    });
    bedrockBuffer = processedChunk.bedrockBuffer;
    return processedChunk.decodedChunk;
  }).join('');
};

/**
 * Handle a chunk of data from the bedrock API.
 * @param chunk - The chunk of data to process.
 * @param bedrockBuffer - The buffer containing the current data.
 * @param chunkHandler - Optional function to handle the chunk once it has been processed.
 * @returns {decodedChunk, bedrockBuffer } - The decoded chunk and the updated buffer.
 */
const handleBedrockChunk = ({
  chunk,
  bedrockBuffer,
  chunkHandler,
  logger
}) => {
  // Concatenate the current chunk to the existing buffer.
  let newBuffer = concatChunks(bedrockBuffer, chunk);
  // Get the length of the next message in the buffer.
  let messageLength = getMessageLength(newBuffer);
  // Initialize an array to store fully formed message chunks.
  const buildChunks = [];
  // Process the buffer until no complete messages are left.
  while (newBuffer.byteLength > 0 && newBuffer.byteLength >= messageLength) {
    // Extract a chunk of the specified length from the buffer.
    const extractedChunk = newBuffer.slice(0, messageLength);
    // Add the extracted chunk to the array of fully formed message chunks.
    buildChunks.push(extractedChunk);
    // Remove the processed chunk from the buffer.
    newBuffer = newBuffer.slice(messageLength);
    // Get the length of the next message in the updated buffer.
    messageLength = getMessageLength(newBuffer);
  }
  const awsDecoder = new _eventstreamCodec.EventStreamCodec(_utilUtf.toUtf8, _utilUtf.fromUtf8);

  // Decode and parse each message chunk, extracting the completion.
  const decodedChunk = buildChunks.map(bChunk => {
    const event = awsDecoder.decode(bChunk);
    const body = JSON.parse(Buffer.from(JSON.parse(new TextDecoder().decode(event.body)).bytes, 'base64').toString());
    const decodedContent = prepareBedrockOutput(body, logger);
    if (chunkHandler) {
      chunkHandler(decodedContent);
    }
    return decodedContent;
  }).join('');
  return {
    decodedChunk,
    bedrockBuffer: newBuffer
  };
};

/**
 * Gets the length of the next message from the buffer.
 *
 * @param {Uint8Array} buffer - Buffer containing the message.
 * @returns {number} - Length of the next message.
 */
exports.handleBedrockChunk = handleBedrockChunk;
function getMessageLength(buffer) {
  // If the buffer is empty, return 0.
  if (buffer.byteLength === 0) return 0;
  // Create a DataView to read the Uint32 value at the beginning of the buffer.
  const view = new DataView(buffer.buffer, buffer.byteOffset, buffer.byteLength);
  // Read and return the Uint32 value (message length).
  return view.getUint32(0, false);
}

/**
 * Concatenates two Uint8Array buffers.
 *
 * @param {Uint8Array} a - First buffer.
 * @param {Uint8Array} b - Second buffer.
 * @returns {Uint8Array} - Concatenated buffer.
 */
function concatChunks(a, b) {
  const newBuffer = new Uint8Array(a.length + b.length);
  // Copy the contents of the first buffer to the new buffer.
  newBuffer.set(a);
  // Copy the contents of the second buffer to the new buffer starting from the end of the first buffer.
  newBuffer.set(b, a.length);
  return newBuffer;
}
/**
 * Prepare the streaming output from the bedrock API
 * @param responseBody
 * @returns string
 */
const prepareBedrockOutput = (responseBody, logger) => {
  if (responseBody.type && responseBody.type.length) {
    var _responseBody$delta, _responseBody$delta2;
    if (responseBody.type === 'message_start' && responseBody.message) {
      return parseContent(responseBody.message.content);
    } else if (responseBody.type === 'content_block_delta' && ((_responseBody$delta = responseBody.delta) === null || _responseBody$delta === void 0 ? void 0 : _responseBody$delta.type) === 'text_delta' && typeof ((_responseBody$delta2 = responseBody.delta) === null || _responseBody$delta2 === void 0 ? void 0 : _responseBody$delta2.text) === 'string') {
      return responseBody.delta.text;
    }
  }
  // ignore any chunks that do not include text output
  return '';
};

/**
 * Parse the content from the bedrock API
 * @param content
 * @returns string
 */
function parseContent(content) {
  let parsedContent = '';
  if (content.length === 1 && content[0].type === 'text' && content[0].text) {
    parsedContent = content[0].text;
  } else if (content.length > 1) {
    // this case should not happen, but here is a fallback
    parsedContent = content.reduce((acc, {
      text
    }) => text ? `${acc}\n${text}` : acc, '');
  }
  return parsedContent;
}