"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.investigationRepositoryFactory = investigationRepositoryFactory;
var _investigationShared = require("@kbn/investigation-shared");
var _investigation = require("../saved_objects/investigation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function investigationRepositoryFactory({
  soClient,
  logger
}) {
  function toInvestigation(stored) {
    const result = _investigationShared.investigationSchema.safeParse(stored);
    if (!result.success) {
      logger.error(`Invalid stored Investigation with id [${stored.id}]`);
      return undefined;
    }
    return result.data;
  }
  function toStoredInvestigation(investigation) {
    return _investigationShared.investigationSchema.parse(investigation);
  }
  return {
    async save(investigation) {
      await soClient.create(_investigation.SO_INVESTIGATION_TYPE, toStoredInvestigation(investigation), {
        id: investigation.id,
        overwrite: true
      });
    },
    async findById(id) {
      const response = await soClient.find({
        type: _investigation.SO_INVESTIGATION_TYPE,
        page: 1,
        perPage: 1,
        filter: `investigation.attributes.id:(${id})`
      });
      if (response.total === 0) {
        throw new Error(`Investigation [${id}] not found`);
      }
      const investigation = toInvestigation(response.saved_objects[0].attributes);
      if (investigation === undefined) {
        throw new Error('Invalid stored Investigation');
      }
      return investigation;
    },
    async deleteById(id) {
      const response = await soClient.find({
        type: _investigation.SO_INVESTIGATION_TYPE,
        page: 1,
        perPage: 1,
        filter: `investigation.attributes.id:(${id})`
      });
      if (response.total === 0) {
        throw new Error(`Investigation [${id}] not found`);
      }
      await soClient.delete(_investigation.SO_INVESTIGATION_TYPE, response.saved_objects[0].id);
    },
    async search({
      search,
      filter,
      pagination
    }) {
      const response = await soClient.find({
        type: _investigation.SO_INVESTIGATION_TYPE,
        page: pagination.page,
        perPage: pagination.perPage,
        sortField: 'updated_at',
        sortOrder: 'desc',
        ...(filter && {
          filter
        }),
        ...(search && {
          search: search.search
        })
      });
      return {
        total: response.total,
        perPage: response.per_page,
        page: response.page,
        results: response.saved_objects.map(savedObject => toInvestigation(savedObject.attributes)).filter(investigation => investigation !== undefined)
      };
    },
    async findAllTags() {
      var _response$aggregation, _response$aggregation2, _response$aggregation3;
      const response = await soClient.find({
        type: _investigation.SO_INVESTIGATION_TYPE,
        aggs: {
          tags: {
            terms: {
              field: 'investigation.attributes.tags',
              size: 10000
            }
          }
        }
      });
      return (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : (_response$aggregation3 = _response$aggregation2.tags) === null || _response$aggregation3 === void 0 ? void 0 : _response$aggregation3.buckets.map(bucket => bucket.key)) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
    },
    async getStats() {
      var _response$aggregation4, _response$aggregation5, _response$aggregation6;
      const response = await soClient.find({
        type: _investigation.SO_INVESTIGATION_TYPE,
        aggs: {
          status: {
            terms: {
              field: 'investigation.attributes.status',
              size: 10
            }
          }
        }
      });
      const countByStatus = {
        active: 0,
        triage: 0,
        mitigated: 0,
        resolved: 0,
        cancelled: 0
      };
      return {
        count: (_response$aggregation4 = (_response$aggregation5 = response.aggregations) === null || _response$aggregation5 === void 0 ? void 0 : (_response$aggregation6 = _response$aggregation5.status) === null || _response$aggregation6 === void 0 ? void 0 : _response$aggregation6.buckets.reduce((acc, bucket) => ({
          ...acc,
          [bucket.key]: bucket.doc_count
        }), countByStatus)) !== null && _response$aggregation4 !== void 0 ? _response$aggregation4 : countByStatus,
        total: response.total
      };
    }
  };
}