"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.confirmConstantsInDataset = confirmConstantsInDataset;
exports.getKeywordAndNumericalFields = getKeywordAndNumericalFields;
exports.getSampleDocuments = getSampleDocuments;
exports.mergeSampleDocumentsWithFieldCaps = mergeSampleDocumentsWithFieldCaps;
exports.sortAndTruncateAnalyzedFields = sortAndTruncateAnalyzedFields;
var _pLimit = _interopRequireDefault(require("p-limit"));
var _lodash = require("lodash");
var _truncate_list = require("@kbn/inference-plugin/common/util/truncate_list");
var _queries = require("./queries");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getSampleDocuments({
  esClient,
  start,
  end,
  indexPatterns,
  count,
  dslFilter
}) {
  return esClient.search({
    index: indexPatterns,
    track_total_hits: true,
    size: count,
    body: {
      query: {
        bool: {
          should: [{
            function_score: {
              functions: [{
                random_score: {}
              }]
            }
          }],
          must: [...(0, _queries.rangeQuery)(start, end), ...(dslFilter !== null && dslFilter !== void 0 ? dslFilter : [])]
        }
      },
      sort: {
        _score: {
          order: 'desc'
        }
      },
      _source: false,
      fields: ['*']
    }
  }).then(response => {
    const hits = response.hits.total;
    if (hits.value === 0) {
      return {
        total: 0,
        samples: []
      };
    }
    return {
      total: hits.value,
      samples: response.hits.hits.map(hit => {
        var _hit$fields;
        return (_hit$fields = hit.fields) !== null && _hit$fields !== void 0 ? _hit$fields : {};
      })
    };
  });
}
async function getKeywordAndNumericalFields({
  indexPatterns,
  esClient,
  start,
  end
}) {
  const fieldCaps = await esClient.fieldCaps({
    index: indexPatterns,
    fields: '*',
    include_empty_fields: false,
    types: ['constant_keyword', 'keyword', 'integer', 'long', 'double', 'float', 'byte', 'boolean', 'alias', 'flattened', 'ip', 'aggregate_metric_double', 'histogram'],
    index_filter: {
      bool: {
        filter: [...(0, _queries.excludeFrozenQuery)(), ...(0, _queries.rangeQuery)(start, end)]
      }
    }
  });
  return Object.entries(fieldCaps.fields).map(([fieldName, fieldSpec]) => {
    return {
      name: fieldName,
      esTypes: Object.keys(fieldSpec)
    };
  });
}
function mergeSampleDocumentsWithFieldCaps({
  total,
  samples,
  fieldCaps
}) {
  const nonEmptyFields = new Set();
  const fieldValues = new Map();
  for (const document of samples) {
    Object.keys(document).forEach(field => {
      var _fieldValues$get;
      if (!nonEmptyFields.has(field)) {
        nonEmptyFields.add(field);
      }
      const values = (0, _lodash.castArray)(document[field]);
      const currentFieldValues = (_fieldValues$get = fieldValues.get(field)) !== null && _fieldValues$get !== void 0 ? _fieldValues$get : [];
      values.forEach(value => {
        if (typeof value === 'string' || typeof value === 'number' || typeof value === 'boolean') {
          currentFieldValues.push(value);
        }
      });
      fieldValues.set(field, currentFieldValues);
    });
  }
  const fields = fieldCaps.flatMap(spec => {
    var _spec$esTypes;
    const values = fieldValues.get(spec.name);
    const countByValues = new Map();
    values === null || values === void 0 ? void 0 : values.forEach(value => {
      var _countByValues$get;
      const currentCount = (_countByValues$get = countByValues.get(value)) !== null && _countByValues$get !== void 0 ? _countByValues$get : 0;
      countByValues.set(value, currentCount + 1);
    });
    const sortedValues = (0, _lodash.sortBy)(Array.from(countByValues.entries()).map(([value, count]) => {
      return {
        value,
        count
      };
    }), 'count', 'desc');
    return {
      name: spec.name,
      types: (_spec$esTypes = spec.esTypes) !== null && _spec$esTypes !== void 0 ? _spec$esTypes : [],
      empty: !nonEmptyFields.has(spec.name),
      cardinality: countByValues.size || null,
      values: (0, _lodash.uniq)(sortedValues.flatMap(({
        value
      }) => value))
    };
  });
  return {
    total,
    sampled: samples.length,
    fields
  };
}
function sortAndTruncateAnalyzedFields(analysis) {
  const {
    fields,
    ...meta
  } = analysis;
  const [nonEmptyFields, emptyFields] = (0, _lodash.partition)(analysis.fields, field => !field.empty);
  const sortedFields = [...(0, _lodash.shuffle)(nonEmptyFields), ...(0, _lodash.shuffle)(emptyFields)];
  return {
    ...meta,
    fields: (0, _truncate_list.truncateList)(sortedFields.map(field => {
      let name = `${field.name}:${field.types.join(',')}`;
      if (field.empty) {
        return `${name} (empty)`;
      }
      name += ` - ${field.cardinality} distinct values`;
      if (field.values.length && (field.types.includes('keyword') || field.types.includes('text')) && field.values.length <= 10) {
        return `${name} (${(0, _truncate_list.truncateList)(field.values.map(value => '`' + value + '`'), field.types.includes('text') ? 2 : 25).join(', ')})`;
      }
      return name;
    }), 500).sort()
  };
}
async function confirmConstantsInDataset({
  esClient,
  constants,
  indexPatterns
}) {
  const limiter = (0, _pLimit.default)(5);
  return Promise.all(constants.map(constant => {
    return limiter(async () => {
      return esClient.termsEnum({
        index: indexPatterns.join(','),
        field: constant.field,
        index_filter: {
          bool: {
            filter: [...(0, _queries.excludeFrozenQuery)()]
          }
        }
      }).then(response => {
        const isConstant = response.terms.length === 1;
        return {
          field: constant.field,
          constant: isConstant,
          value: isConstant ? response.terms[0] : undefined
        };
      });
    });
  }));
}