"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.combineProcessors = combineProcessors;
exports.createCSVProcessor = createCSVProcessor;
exports.createDropProcessor = createDropProcessor;
exports.createGrokProcessor = createGrokProcessor;
exports.createKVProcessor = createKVProcessor;
exports.createPassthroughFailureProcessor = createPassthroughFailureProcessor;
exports.createRemoveProcessor = createRemoveProcessor;
var _jsYaml = require("js-yaml");
var _path = require("path");
var _nunjucks = require("nunjucks");
var _util = require("./util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function combineProcessors(initialPipeline, processors) {
  // Create a deep copy of the initialPipeline to avoid modifying the original input
  const currentPipeline = (0, _util.deepCopy)(initialPipeline);
  if (Object.keys(processors === null || processors === void 0 ? void 0 : processors.processors).length === 0) {
    return currentPipeline;
  }
  // Add the new processors right before the last 2 remove processor in the initial pipeline.
  // This is so all the processors if conditions are not accessing possibly removed fields.
  const currentProcessors = currentPipeline.processors;
  const appendProcessors = createAppendProcessors(processors);
  const combinedProcessors = [...currentProcessors.slice(0, -2), ...appendProcessors, ...currentProcessors.slice(-2)];
  currentPipeline.processors = combinedProcessors;
  return currentPipeline;
}

// The related and categorization graphs returns a simplified array of append processors.
// This function converts the simplified array to the full ESProcessorItem array.
function createAppendProcessors(processors) {
  const templatesPath = (0, _path.join)(__dirname, '../templates/processors');
  const env = new _nunjucks.Environment(new _nunjucks.FileSystemLoader(templatesPath), {
    autoescape: false
  });
  const template = env.getTemplate('append.yml.njk');
  const renderedTemplate = template.render({
    processors
  });
  const appendProcessors = (0, _jsYaml.safeLoad)(renderedTemplate);
  return appendProcessors;
}

// The kv graph returns a simplified grok processor for header
// This function takes in the grok pattern string and creates the grok processor
function createGrokProcessor(grokPatterns) {
  const templatesPath = (0, _path.join)(__dirname, '../templates/processors');
  const env = new _nunjucks.Environment(new _nunjucks.FileSystemLoader(templatesPath), {
    autoescape: false
  });
  const template = env.getTemplate('grok.yml.njk');
  const renderedTemplate = template.render({
    grokPatterns
  });
  const grokProcessor = (0, _jsYaml.safeLoad)(renderedTemplate);
  return grokProcessor;
}

// The kv graph returns a simplified kv processor for structured body
// This function takes in the kvInput string and creates the kv processor
function createKVProcessor(kvInput, state) {
  const templatesPath = (0, _path.join)(__dirname, '../templates/processors');
  const env = new _nunjucks.Environment(new _nunjucks.FileSystemLoader(templatesPath), {
    autoescape: false
  });
  const template = env.getTemplate('kv.yml.njk');
  const renderedTemplate = template.render({
    kvInput,
    packageName: state.packageName,
    dataStreamName: state.dataStreamName
  });
  const kvProcessor = (0, _jsYaml.safeLoad)(renderedTemplate);
  return kvProcessor;
}

// Processor for the csv input to convert it to JSON.
function createCSVProcessor(source, targets) {
  return {
    csv: {
      field: source,
      target_fields: targets,
      description: 'Parse CSV input',
      tag: 'parse_csv'
    }
  };
}

// Trivial processor for the on_failure part of the pipeline.
// Use only if the source of error is not necessary.
function createPassthroughFailureProcessor() {
  return {
    append: {
      field: 'error.message',
      description: 'Append the error message as-is',
      tag: 'append_error_message',
      value: '{{{_ingest.on_failure_message}}}'
    }
  };
}

// Processor to remove the message field.
function createRemoveProcessor() {
  return {
    remove: {
      field: 'message',
      ignore_missing: true,
      description: 'Remove the message field',
      tag: 'remove_message_field'
    }
  };
}

// Processor to drop the specific values.
// values is a record of key value pairs to match against the fields
// root is the root of the fields to match against
function createDropProcessor(values, prefix, tag, description) {
  const prefixExpression = prefix.join('?.');
  const conditions = Object.entries(values).map(([key, value]) => `ctx.${prefixExpression}?.${key} == '${String(value)}'`).join(' && ');
  return {
    drop: {
      if: conditions,
      ignore_failure: true,
      description,
      tag
    }
  };
}