"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAnalyzeLogsRoutes = registerAnalyzeLogsRoutes;
var _server = require("@kbn/data-plugin/server");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _common = require("../../common");
var _constants = require("../constants");
var _graph = require("../graphs/log_type_detection/graph");
var _llm = require("../util/llm");
var _route_validation = require("../util/route_validation");
var _with_availability = require("./with_availability");
var _errors = require("../lib/errors");
var _routes_util = require("./routes_util");
var _constants2 = require("../../common/constants");
var _cef_error = require("../lib/errors/cef_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerAnalyzeLogsRoutes(router) {
  router.versioned.post({
    path: _common.ANALYZE_LOGS_PATH,
    access: 'internal',
    options: {
      timeout: {
        idleSocket: _constants.ROUTE_HANDLER_TIMEOUT
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidationWithZod)(_common.AnalyzeLogsRequestBody)
      }
    }
  }, (0, _with_availability.withAvailability)(async (context, req, res) => {
    const {
      packageName,
      dataStreamName,
      packageTitle,
      dataStreamTitle,
      logSamples,
      langSmithOptions
    } = req.body;
    const services = await context.resolve(['core']);
    const {
      client
    } = services.core.elasticsearch;
    const {
      getStartServices,
      logger
    } = await context.integrationAssistant;
    const [, {
      actions: actionsPlugin
    }] = await getStartServices();
    try {
      var _connector$config, _langSmithOptions$pro;
      const actionsClient = await actionsPlugin.getActionsClientWithRequest(req);
      const connector = await actionsClient.get({
        id: req.body.connectorId
      });
      const abortSignal = (0, _server.getRequestAbortedSignal)(req.events.aborted$);
      const actionTypeId = connector.actionTypeId;
      const llmType = (0, _llm.getLLMType)(actionTypeId);
      const llmClass = (0, _llm.getLLMClass)(llmType);
      const model = new llmClass({
        actionsClient,
        connectorId: connector.id,
        logger,
        llmType,
        model: (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : _connector$config.defaultModel,
        temperature: 0.05,
        maxTokens: 4096,
        signal: abortSignal,
        streaming: false
      });
      const options = {
        callbacks: [new _apm.APMTracer({
          projectName: (_langSmithOptions$pro = langSmithOptions === null || langSmithOptions === void 0 ? void 0 : langSmithOptions.projectName) !== null && _langSmithOptions$pro !== void 0 ? _langSmithOptions$pro : 'default'
        }, logger), ...(0, _langsmith.getLangSmithTracer)({
          ...langSmithOptions,
          logger
        })]
      };
      const logFormatParameters = {
        packageName,
        dataStreamName,
        packageTitle,
        dataStreamTitle,
        logSamples
      };
      const graph = await (0, _graph.getLogFormatDetectionGraph)({
        model,
        client
      });
      const graphResults = await graph.invoke(logFormatParameters, options);
      const graphLogFormat = graphResults.results.samplesFormat.name;
      switch (graphLogFormat) {
        case 'unsupported':
          throw new _errors.UnsupportedLogFormatError({
            message: _constants2.GenerationErrorCode.UNSUPPORTED_LOG_SAMPLES_FORMAT
          });
        case 'cef':
          throw new _cef_error.CefError(_constants2.GenerationErrorCode.CEF_ERROR);
        case 'leef':
          throw new _errors.UnsupportedLogFormatError({
            message: _constants2.GenerationErrorCode.UNSUPPORTED_LOG_SAMPLES_FORMAT,
            logFormat: 'Log Event Extended Format (LEEF)'
          });
        case 'fix':
          throw new _errors.UnsupportedLogFormatError({
            message: _constants2.GenerationErrorCode.UNSUPPORTED_LOG_SAMPLES_FORMAT,
            logFormat: 'Financial Information eXchange (FIX)'
          });
      }
      return res.ok({
        body: _common.AnalyzeLogsResponse.parse(graphResults)
      });
    } catch (err) {
      try {
        (0, _routes_util.handleCustomErrors)(err, _constants2.GenerationErrorCode.RECURSION_LIMIT_ANALYZE_LOGS);
      } catch (e) {
        if ((0, _errors.isErrorThatHandlesItsOwnResponse)(e)) {
          return e.sendResponse(res);
        }
      }
      return res.badRequest({
        body: err
      });
    }
  }));
}