"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEcsGraph = getEcsGraph;
exports.getEcsSubGraph = getEcsSubGraph;
var _langgraph = require("@langchain/langgraph");
var _duplicates = require("./duplicates");
var _invalid = require("./invalid");
var _mapping = require("./mapping");
var _missing = require("./missing");
var _model = require("./model");
var _state = require("./state");
var _validate = require("./validate");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const handleCreateMappingChunks = async ({
  state
}) => {
  // Cherrypick a shallow copy of state to pass to subgraph
  const stateParams = {
    exAnswer: state.exAnswer,
    prefixedSamples: state.prefixedSamples,
    ecs: state.ecs,
    dataStreamName: state.dataStreamName,
    packageName: state.packageName,
    samplesFormat: state.samplesFormat,
    additionalProcessors: state.additionalProcessors
  };
  if (Object.keys(state.currentMapping).length === 0) {
    return state.sampleChunks.map(chunk => {
      return new _langgraph.Send('subGraph', {
        ...stateParams,
        combinedSamples: chunk
      });
    });
  }
  return 'modelOutput';
};
function chainRouter({
  state
}) {
  if (Object.keys(state.finalMapping).length === 0 && state.hasTriedOnce) {
    return 'modelOutput';
  }
  if (Object.keys(state.duplicateFields).length > 0) {
    return 'duplicateFields';
  }
  if (Object.keys(state.missingKeys).length > 0) {
    return 'missingKeys';
  }
  if (Object.keys(state.invalidEcsFields).length > 0) {
    return 'invalidEcsFields';
  }
  if (!state.finalized) {
    return 'modelOutput';
  }
  return _langgraph.END;
}

// This is added as a separate graph to be able to run these steps concurrently from handleCreateMappingChunks
async function getEcsSubGraph({
  model
}) {
  const workflow = new _langgraph.StateGraph({
    channels: _state.graphState
  }).addNode('modelSubOutput', state => (0, _model.modelSubOutput)({
    state
  })).addNode('handleValidation', state => (0, _validate.handleValidateMappings)({
    state
  })).addNode('handleEcsMapping', state => (0, _mapping.handleEcsMapping)({
    state,
    model
  })).addNode('handleDuplicates', state => (0, _duplicates.handleDuplicates)({
    state,
    model
  })).addNode('handleMissingKeys', state => (0, _missing.handleMissingKeys)({
    state,
    model
  })).addNode('handleInvalidEcs', state => (0, _invalid.handleInvalidEcs)({
    state,
    model
  })).addEdge(_langgraph.START, 'handleEcsMapping').addEdge('handleEcsMapping', 'handleValidation').addEdge('handleDuplicates', 'handleValidation').addEdge('handleMissingKeys', 'handleValidation').addEdge('handleInvalidEcs', 'handleValidation').addConditionalEdges('handleValidation', state => chainRouter({
    state
  }), {
    duplicateFields: 'handleDuplicates',
    missingKeys: 'handleMissingKeys',
    invalidEcsFields: 'handleInvalidEcs',
    modelOutput: 'modelSubOutput'
  }).addEdge('modelSubOutput', _langgraph.END);
  const compiledEcsSubGraph = workflow.compile().withConfig({
    runName: 'ECS Mapping (Chunk)'
  });
  return compiledEcsSubGraph;
}
async function getEcsGraph({
  model
}) {
  const subGraph = await getEcsSubGraph({
    model
  });
  const workflow = new _langgraph.StateGraph({
    channels: _state.graphState
  }).addNode('modelInput', state => (0, _model.modelInput)({
    state
  })).addNode('modelOutput', state => (0, _model.modelOutput)({
    state
  })).addNode('handleValidation', state => (0, _validate.handleValidateMappings)({
    state
  })).addNode('handleDuplicates', state => (0, _duplicates.handleDuplicates)({
    state,
    model
  })).addNode('handleMissingKeys', state => (0, _missing.handleMissingKeys)({
    state,
    model
  })).addNode('handleInvalidEcs', state => (0, _invalid.handleInvalidEcs)({
    state,
    model
  })).addNode('handleMergedSubGraphResponse', state => (0, _model.modelMergedInputFromSubGraph)({
    state
  })).addNode('subGraph', subGraph).addEdge(_langgraph.START, 'modelInput').addEdge('subGraph', 'handleMergedSubGraphResponse').addEdge('handleDuplicates', 'handleValidation').addEdge('handleMissingKeys', 'handleValidation').addEdge('handleInvalidEcs', 'handleValidation').addEdge('handleMergedSubGraphResponse', 'handleValidation').addConditionalEdges('modelInput', state => handleCreateMappingChunks({
    state
  }), {
    modelOutput: 'modelOutput',
    subGraph: 'subGraph'
  }).addConditionalEdges('handleValidation', state => chainRouter({
    state
  }), {
    duplicateFields: 'handleDuplicates',
    missingKeys: 'handleMissingKeys',
    invalidEcsFields: 'handleInvalidEcs',
    modelOutput: 'modelOutput'
  }).addEdge('modelOutput', _langgraph.END);
  const compiledEcsGraph = workflow.compile().withConfig({
    runName: 'ECS Mapping'
  });
  return compiledEcsGraph;
}